
if bhv_tree then
	return
end

behavior_trees = {} -- the behavior trees for all running assemblies are stored here
bhv_tree_node = {}  -- common functions for all nodes
bhv_tree = {}       -- a table with factory functions
bhv_tree.mt = {__index = bhv_tree_node} -- inherit the functions in the table bhv_tree_node

-- constants
bhv_tree.invalid = 1
bhv_tree.failure = 2
bhv_tree.running = 3
bhv_tree.success = 4

-- TODO: Repeater, Repeat Until Fail, Repeat Until Succeed ?

-- Creates a generic behavior tree node. The user must define an update function
function bhv_tree:new_node(id)
	local members = {}
	if not id then
		print("ERROR: bhv_tree:new_node needs a unique identifier as argument.")
		return
	end
	
	members.id = id
	function members:update(assy)
		print("ERROR: bhv_tree:new_node "..self.id..", no custom update function defined.")
		return bhv_tree.invalid
	end
	
	return setmetatable(members, bhv_tree.mt)
end

-- Creates a sequencer node: executes all child behaviours and returns to the parent node if any fail
function bhv_tree:new_sequencer()
	local members = {}
	function members:update(assy)
		for _, bhv in ipairs(self.children) do
			local state = bhv:update(assy)
			if state ~= bhv_tree.success then
				return state
			end
		end
		return bhv_tree.success
	end
	
	return setmetatable(members, bhv_tree.mt)
end

-- Creates a selector node: executes all child behaviours and returns to the parent node at the first success
function bhv_tree:new_selector()
	local members = {}
	function members:update(assy)
		for _, bhv in ipairs(self.children) do
			local state = bhv:update(assy)
			if state ~= bhv_tree.failure then
				return state
			end
		end
		return bhv_tree.failure
	end
	
	return setmetatable(members, bhv_tree.mt)
end

-- Creates a inverter node: run the only child and return "failure" if the child succeeded and "success" if it fails
function bhv_tree:new_inverter()
	local members = {}
	function members:update(assy)
		if self.children then
			local state = self.children[1]:update(assy)
			if state == bhv_tree.success then
				return bhv_tree.failure
			elseif state == bhv_tree.failure then
				return bhv_tree.success
			else
				return state
			end
		end
		return bhv_tree.invalid
	end
	
	return setmetatable(members, bhv_tree.mt)
end

-- Creates a succeeder node: run the only child and always return success (ignores what the child returned)
function bhv_tree:new_succeeder()
	local members = {}	
	function members:update(assy)
		if self.children then
			self.children[1]:update(assy)
			return bhv_tree.success
		end
		return bhv_tree.invalid
	end
	
	return setmetatable(members, bhv_tree.mt)
end

-- Adds a child to any node
function bhv_tree_node:add_child(child)
	if not self.children then
		self.children = {}
	end
	table.insert(self.children, child)
end

-- Returns a pointer to an assembly specific storage table that is used for storing the state of this node. If no table exist it will be created and its state set to invalid.
function bhv_tree_node:get_context(assembly)
	if not assembly.bt_context then
		assembly.bt_context = {}
	end
	
	if not assembly.bt_context[self.id] then
		assembly.bt_context[self.id] = {}
		assembly.bt_context[self.id].state = bhv_tree.invalid
	end
	
	return assembly.bt_context[self.id]
end