
if not pioneer_std then pioneer_std = {} end

-- Pioneer standardized Build. This is called at beginning of Pioneer Build in its own script
function pioneer_std.build(pio)
  if not pio then return end

  ------------------------------------------------
  -- Init Constants

  ---------------------------------------------------
  -- Pose Animations Enum
  -- TODO: Get these directly from the blueprint, where they are defined in BP pose editor?

  pio.poseanim = {}
  pio.poseanim.none = { name = "None", count = 1, interval = 1.0, loop = false }
  pio.poseanim.idle = { name = "Idle", count = 1, interval = 0, loop = true }
  pio.poseanim.die = { name = "Die", count = 1, interval = 0.3, loop = false }
  pio.poseanim.slack = { name = "Slack", count = 1, interval = 0.3, loop = false }
  pio.poseanim.ride = { name = "Ride", count = 1, interval = 0.3, loop = false }
  pio.poseanim.step = { name = "Step", count = 3, interval = 0.15, loop = false }
  pio.poseanim.walk = { name = "Walk", count = 6, interval = 0.15, loop = true }
  pio.poseanim.run = { name = "Run", count = 6, interval = 0.075, loop = true }
  pio.poseanim.climb = { name = "Climb", count = 6, interval = 0.15, loop = true }
  pio.poseanim.crawl = { name = "Crawl", count = 6, interval = 0.075, loop = true }
  pio.poseanim.prejump = { name = "PreJump", count = 1, interval = 0.50, loop = false }
  pio.poseanim.jump = { name = "Jump", count = 1, interval = 0.25, loop = false }
  pio.poseanim.postjump = { name = "PostJump", count = 1, interval = 0.25, loop = false }
  pio.poseanim.fall = { name = "Fall", count = 4, interval = 0.15, loop = true }
  pio.poseanim.fly = { name = "Fly", count = 1, interval = 0.25, loop = false }
  pio.poseanim.get_up_back = { name = "Get Up Back", count = 1, interval = 0.20, loop = false }
  pio.poseanim.get_up_front = { name = "Get Up Front", count = 1, interval = 0.20, loop = false }
  pio.poseanim.go_prone = { name = "Go Prone", count = 2, interval = 0.25, loop = false }
  pio.poseanim.teeter = { name = "Teeter", count = 3, interval = 0.15, loop = true }
  pio.poseanim.skid = { name = "Skid", count = 1, interval = 0.5, loop = false }
  pio.poseanim.slide = { name = "Slide", count = 1, interval = 0.05, loop = true }
  pio.poseanim.grab_tool = { name = "Grab Tool", count = 1, interval = 0.25, loop = false }
  pio.poseanim.draw_tool = { name = "Draw Tool", count = 1, interval = 0.25, loop = false }
  pio.poseanim.mounted = { name = "Mounted", count = 1, interval = 0.25, loop = false }
  pio.poseanim.point_at = { name = "Point At", count = 1, interval = 0.15, loop = false }

  ---------------------------------------------------
  -- Joint Group State Enumeration
  
  -- Temporary name list we'll use to create the state table
  local joint_group_state_names = {
  "IDLE",
  "DIE",
  "SLACK",
  "BRACE",
  "RIDE",
  "STEP",
  "WALK",
  "RUN",
  "CLIMB",
  "CRAWL",
  "GET_UP_BACK",
  "GET_UP_FRONT",
  "GO_PRONE",
  "TEETER",
  "PREJUMP",
  "JUMP",
  "POSTJUMP",
  "FALL",
  "FLY",
  "SKID",
  "SLIDE",
  "SHIELD",
  "GRAB_TOOL",
  "DRAW_TOOL",
  "AIM_TOOL_AT",
  "POINT_AT",
  "ATOMIZE",
  "ASSEMBLE"
  }

  -- Create the state table for Joint Group States
  pio.jgs = {}
  for i, v in ipairs(joint_group_state_names) do
    pio.jgs[v] = i
  end

  ---------------------------------------------------
  -- Init State Variables

  -- Joint Groups
  pio.jg = {}
  --pio.jg.all = { name = "", state = pio.jgs.IDLE, state_changed = true, anim = pio.poseanim.idle, anim_num = 1, anim_speed = 1.0 }
  pio.jg.legs = { name = "Leg", state = pio.jgs.IDLE, state_changed = true,  anim = pio.poseanim.idle, anim_num = 1, anim_speed = 1.0, hold_timer = pio:CreateTimer(), point_target = VectorF }
  pio.jg.arm_fg = { name = "Arm FG", state = pio.jgs.IDLE, state_changed = true, anim = pio.poseanim.idle, anim_num = 1, anim_speed = 1.0, hold_timer = pio:CreateTimer(), point_target = VectorF }
  pio.jg.arm_bg = { name = "Arm BG", state = pio.jgs.IDLE, state_changed = true, anim = pio.poseanim.idle, anim_num = 1, anim_speed = 1.0, hold_timer = pio:CreateTimer(), point_target = VectorF }
  pio.jg.neck = { name = "Neck", state = pio.jgs.IDLE, state_changed = true, anim = pio.poseanim.idle, anim_num = 1, anim_speed = 1.0, hold_timer = pio:CreateTimer(), point_target = VectorF }

  -- Balance states
  pio.balance = {}
  pio.balance.NONE = 1
  pio.balance.BIPED = 2
  pio.balance.GET_UP = 3
  pio.balance.GO_PRONE = 4
  pio.balance.FLYING = 5
  pio.balance.PRONE = 6

  -- Misc state things

  -- Relative or Absolute flight control mode
-- TODO: Make this set by a player-specific setting the engine gives us
  pio.relative_flight_control = c2d_settings.relative_flight_control
  pio.is_ambulating_left = false
  pio.ambulation_input = VectorF(0, 0)
  pio.jetpack_input = 0
  pio.sliding_input = 0
	-- Last recorded Aurium Store value in the current Controller
  pio.aurium_last_store = 0
  -- Current suit Aurium charge level
  pio.aurium_charge = 0
  -- Max capacity of the Aurium battery/capacitor
  pio.aurium_capacity = 100
  -- How much aurium per second is recharged from Aurium store into the battery
  pio.aurium_recharge_rate = 50
  -- How much of a delay (in s) after all Aurium consumption stops until recharge starts
  pio.aurium_recharge_delay = 0.5
	-- How much A is consumed each s of flight
	pio.flight_aurium_consumption = 150
  
  -- Current balancing state
  pio.balance_state = pio.balance.NONE
  -- The lean of the torso from angle 0 that the balancer tries to achieve
  pio.balance_lean = 0
  pio.balance_power = 6000
  pio.slippery_limbs = 0.5
  pio.slippery_body = 0.5
  -- Raycasting vectors
  pio.raycast_legvec_right = VectorF(0.6, 0.0)
  pio.raycast_legvec_left = VectorF(-0.6, 0.0)
  pio.arm_fg_point_at_target = VectorF(0, 0)
  pio.arm_bg_point_at_target = VectorF(0, 0)
  pio.neck_look_at_target = VectorF(0, 0)
  pio.tool_aim_at_target = VectorF(0, 0)
  -- Tool Mounting
  -- little radar for searching for tools nearby
  pio.tool_radar = pio:GetRadars():AddRadar("Tool Search")
  pio.tool_radar:SetEnabled(Yes)
  pio.tool_radar:SetUpCone(VectorF(0,0), 0, math.pi*2, 1.5)
  pio.tool_radar:SetUpCheck(Yes, No, No, No)
  -- Vehicle Mounting
  -- here we'll store id of vehicle assembly to which we're currently entering
  pio.entering_vehicle_aid = -1
  -- here we'll store id of vehicle assembly from which we're currently escaping, so we can make pioneer automatically walk left or right away from vehicle
  pio.escaping_vehicle_aid = -1
  -- if we press to escape vehicle, we need to ignore the enter/exit button pressed then until released that button
  pio.ignore_enter_vehicle_now = No

  -- Commonly accessed MOs
  pio.head_mo = pio:GetMOByName("Head")
  pio.body_mo = pio:GetMOByName("Torso")
  pio.hip_joint = pio:GetJointByName("Leg FG Hip")
  if pio.hip_joint == nil then pio.hip_joint = pio:GetJointByName("Leg BG Hip") end

  ----------------------------------------------------
  -- Timers

  -- How long ago we changed mirroring
  pio.mirror_change_timer = pio:CreateTimer()
  
  -- How long ago we changed aim point
  pio.aim_point_change_timer = pio:CreateTimer()
  
  -- How long we are dying
  pio.dying_timer = pio:CreateTimer()
  pio.dying_timer:SetTimeLimit(1.0)

	-- Aurium charge timers
	-- How long since nothing used any aurium at all
	pio.no_aurium_use_timer = pio:CreateTimer()
	pio.no_aurium_use_timer:SetTimeLimit(pio.aurium_recharge_delay)
	pio.aurium_charge_change_timer = pio:CreateTimer()
	pio.aurium_recharge_timer = pio:CreateTimer()
	-- Time since last time this' controller's A store changed value
	pio.aurium_store_change_timer = pio:CreateTimer()

  -- Auto fire timer
  pio.last_fire_timer = pio:CreateTimer()
  pio.last_fire_timer:SetTimeLimit(0.1)
  
  -- Timer since last time we lost balance
  pio.regain_balance_timer = pio:CreateTimer()
  -- How long it takes to fully regain balance after losing it (and being able to regain)
  pio.regain_balance_timer:SetTimeLimit(0.5)
  -- Start off in balanced state
  pio.regain_balance_timer:SetTime(1.1)
  -- Timer since we started climbing
  pio.climbing_timer = pio:CreateTimer()
  pio.climbing_timer:SetTimer(0, 0.75)
  -- Timer since we started flying this burst
  pio.total_flight_timer = pio:CreateTimer()
  -- Timer since we stopped giving flight input
  pio.flight_stop_timer = pio:CreateTimer()
  -- Timer for checking how long our feet have both been on one side of CoM
  pio.off_balance_timer = pio:CreateTimer()
  -- Timer for shifting weight when skidding to a stop after walking
  pio.counter_lean_timer = pio:CreateTimer()
  -- Timer for looking for tool nearby
  pio.look_for_tool_timer = pio:CreateTimer()
  pio.look_for_tool_timer:SetTimer(0, 1)

  -----------------------------------------------------
  -- Physics Setup

  -- for now we deny deformations on limbs completely
  pio:MOSet("Leg"):SetAllowPlasticDeform(false)
  pio:MOSet("Arm"):SetAllowPlasticDeform(false)

  -----------------------------------------------------
  -- Joints Setup

  -- commonly used spring powers
  pio.angular_spring_power_stiff = 6000
  pio.angular_spring_power_stiffer = 7000
  pio.angular_spring_power_loose = 1000
  pio.angular_spring_friction = 40
  
  pio:SetAngularSpringPowers("Leg", pio.angular_spring_power_stiff)
  pio:SetAngularSpringPowers("Arm", pio.angular_spring_power_stiff)
  pio:SetAngularSpringPowers("Neck", pio.angular_spring_power_stiff)
  pio:SetAngularFrictions("Leg", pio.angular_spring_friction)
  pio:SetAngularFrictions("Arm", pio.angular_spring_friction)
  pio:SetAngularFrictions("Neck", pio.angular_spring_friction)

  ------------------------------------------------------
  -- Balancer Setup

  -- so far we use pio simple magic balancer to keep pioneer on legs
  pio:MOSet("Head"):SetBalancer(false)
  pio:MOSet("Head"):SetBalancerPower(pio.balance_power * 0.80)
  pio:MOSet("Torso"):SetBalancer(false)
  pio:MOSet("Torso"):SetBalancerPower(pio.balance_power)

  ---------------------------------------------------------
  -- Particle Systems Setup

  -- switch off jet particles
  --pio:GetPSByName("Jetpack"):SwitchOFFParticleEmitting()

  --------------------------------------------------------
  -- Mount to vehicle if pio starts close to one
  
  local entering_vehicle_assy = scene:GetNearestAssemblyWithFreeMountJointByAT(pio, PP.type_vehicle, 6)
  if entering_vehicle_assy then 
    local mount_joint = entering_vehicle_assy:GetNearestAndFreeMountJoint(pio:GetAssemblyApproxPos())
    local pos = mount_joint:GetWorldPositionA()
    pio:MOSet(""):SetPositionAndOrientation(pos, scene:GetGravityRotAngleAtPoint(pos))
    
    pio:MountToJoint(mount_joint, pio.body_mo, VectorF(0,0), Yes)
    -- the which parameter will infrom vehicle from which mount joint the pioneer leaves the vehicle (because more pioneers can be mounted to one vehicle)
    local which = mount_joint:GetName()
    local vehicle_obj = GetAssemblyLuaObj(entering_vehicle_assy)
    if vehicle_obj then 
      if vehicle_obj.OnVehiclePilotEnter then
        vehicle_obj.OnVehiclePilotEnter(which)
      end
    end
  end


  --------------------------------------------------------
  -- CONTROLLER INPUT HANDLING
  --------------------------------------------------------
   
  function pio.handle_input(controller)

    if not (controller and pio.head_mo and pio.body_mo) or pio.is_dying() or pio:IsDead() then
      return
    end

    -------- MOUNTING EXAMPLE CODE ----------  
    -- update tool radar pos
    if pio.head_mo then
      pio.tool_radar:SetConePoint(pio.head_mo:GetPosition())
    end
    
    -- throw away / pick up tool  
    if controller:Btn_PrevTool_Press() then
      local unmounted_now = No 
      local tool_assy = pio:GetAssemblyMountedToJoint("Tool Mount Arm BG")
      if not tool_assy then tool_assy = pio:GetAssemblyMountedToJoint("Tool Mount Arm FG") end
      if tool_assy then
        -- throw away the tool
        if tool_assy then
          local tool_ass_obj = GetAssemblyLuaObj(tool_assy)
          if tool_ass_obj then 
            if tool_ass_obj.OnToolDeactivate then 
              tool_ass_obj.OnToolDeactivate() 
            end
          end
          pio:UnmountAssembly(tool_assy)
          pio:GetController():SetCanAtomize(true)
          pio:GetController():SetCanAssemble(true)
        end
        unmounted_now = Yes
      end
      if not unmounted_now then
        -- set time to look for tool to grab
        pio.look_for_tool_timer:Reset()
        pio.look_for_tool_timer:Play()
      end
    end
    
    if pio.look_for_tool_timer:GetPlaying() then
      -- pick up tool
      if (pio.tool_radar:GetSignalsCount()>0) then
        if pio:GetMountJointIsMounted("Tool Mount Arm FG") then
          pio.look_for_tool_timer:Pause()
        else
          local mo = pio.tool_radar:GetSignal(0):GetMO()
          if mo then
            -- there is some MO on the radar signal
            local found_assy = scene:GetMOLinkedAssembly(mo) -- ToAssembly(mo:GetAssemblyLinked())
            if found_assy then
              
              -- TOOL PICKUP
              -- check if the assembly type is "Tool"
              if found_assy:GetAT() == "Tool" then
                -- it's a tool, let's mount it to pioneer 
                if found_assy then
                  if not found_assy:GetMountedToAssemblyByAT(PP.type_pioneer) then --if not found_assy:IsMountedToAnything() then 
                    -- we need to get tool assembly's MO that we will mount to pioneer
                    local tool_mo = found_assy:GetMasterMO() -- we choose master MO of tool assembly
                    if tool_mo then
                      -- we'll maybe need to mirror the tool, let's see
                      if found_assy:IsMirrored() ~= pio:IsMirrored() then
                        found_assy:Mirror(Yes)
                      end
                      
                      --pio:MountMOByJointNames(tool_mo, "Tool Mount Back")
                      if pio:MountMOByJointNames(tool_mo, "Tool Mount Arm BG", Yes) 
                      and pio:MountMOByJointNames(tool_mo, "Tool Mount Arm FG", Yes) then
                        pio:GetController():SetCanAtomize(false)
                        pio:GetController():SetCanAssemble(false)
                        pio.look_for_tool_timer:Pause()
                        local found_assy_obj = GetAssemblyLuaObj(found_assy)
                        if found_assy_obj then 
                          if found_assy_obj.OnToolActivate then 
                            found_assy_obj.OnToolActivate() 
                          end
                        end
                      end
                    end
                  end
                end
                
              -- INGOT PICKUP
              elseif found_assy:GetAT() == "Ingot" then
                -- it's an ingot, let's mount it to pioneer, and pioneer power-cleans it above head
                if found_assy then
                  if not found_assy:GetMountedToAssemblyByAT(PP.type_pioneer) then --if not found_assy:IsMountedToAnything() then 
                    -- we need to get tool assembly's MO that we will mount to pioneer
                    local tool_mo = found_assy:GetMasterMO() -- we choose master MO of tool assembly
                    if tool_mo then
                      -- we'll maybe need to mirror the tool, let's see
                      if found_assy:IsMirrored() ~= pio:IsMirrored() then
                        found_assy:Mirror(Yes)
                      end
                      
                      if pio:MountMOByJointNames(tool_mo, "Tool Mount Arm BG", Yes) 
                      and pio:MountMOByJointNames(tool_mo, "Tool Mount Arm FG", Yes) then
                        pio:GetController():SetCanAtomize(false)
                        pio:GetController():SetCanAssemble(false)
                        pio.look_for_tool_timer:Pause()
                        local found_assy_obj = GetAssemblyLuaObj(found_assy)
                        if found_assy_obj then 
                          if found_assy_obj.OnToolActivate then 
                            found_assy_obj.OnToolActivate() 
                          end
                        end
                      end
                    end
                  end
                end
              end
            end
          end
        end
      end
    end
    
--[[ This is now used for Jetpack activation
    
    if controller:Btn_NextTool_Press() then
      -- put mounted tool from arms to back
      local something_in_hands = pio:GetMountJointIsMounted("Tool Mount Arm FG")
      local something_on_back = pio:GetMountJointIsMounted("Tool Mount Back")

      if something_in_hands and not something_on_back then
        -- unmount tool from arms, this equals to: pio:UnmountAssembly(pio.tool_mounted_to_arms)
        local found_assy = pio:GetAssemblyMountedToJoint("Tool Mount Arm FG")
        if found_assy then
          -- unmount tool
          pio:UnmountAssembly(found_assy)
          -- and mount it on back
          pio:MountMOByJointNames(found_assy:GetMasterMO(), "Tool Mount Back", Yes)
          -- deactivate tool
          local found_assy_obj = GetAssemblyLuaObj(found_assy)
          if found_assy_obj then
            if found_assy_obj.OnToolDeactivate then 
              found_assy_obj.OnToolDeactivate() 
            end
          end
          pio:GetController():SetCanAtomize(true)  
          pio:GetController():SetCanAssemble(true)
        end
      else 
        if something_on_back and not something_in_hands then
          -- get mounted tool from back to arms
          local found_assy = pio:GetAssemblyMountedToJoint("Tool Mount Back")
          if found_assy then
            -- unmount tool
            pio:UnmountAssembly(found_assy)
            -- and mount it to hands
            pio:MountMOByJointNames(found_assy:GetMasterMO(), "Tool Mount Arm BG", Yes)
            pio:MountMOByJointNames(found_assy:GetMasterMO(), "Tool Mount Arm FG", Yes)
            -- activate tool
            local found_assy_obj = GetAssemblyLuaObj(found_assy)
            if found_assy_obj then 
              if found_assy_obj.OnToolActivate then 
                found_assy_obj.OnToolActivate() 
              end
            end
            pio:GetController():SetCanAtomize(false)
            pio:GetController():SetCanAssemble(false)
          end
        end 
      end
    end
]]--

  -------- end of MOUNTING EXAMPLE CODE ----------  


  ----------- NEW TOOL CONTROL EXAMPLE --------------------
    
    -- controling the tool, shooting, aiming, etc.
    -- first let's find tool that is ind hands
    local tool_assy = pio:GetAssemblyMountedToJoint("Tool Mount Arm BG")
    if tool_assy then
      local tool_obj = GetAssemblyLuaObj(tool_assy)
      if tool_obj then 
      
        -- send events to tool assembly by controller input
        
        -- first send aim point, then the clicks
        if tool_obj.ToolAimPoint then
          tool_obj.ToolAimPoint(controller:AimPoint())
        end
        
        if controller:Btn_Tool1_Press() then
          if tool_obj.ToolLeftButtonPress then
            tool_obj.ToolLeftButtonPress()
          end
        end
        if controller:Btn_Tool1() then
          if tool_obj.ToolLeftButtonHold then
            tool_obj.ToolLeftButtonHold()
          end
        end
        if controller:Btn_Tool1_Release() then
          if tool_obj.ToolLeftButtonRelease then
            tool_obj.ToolLeftButtonRelease()
          end
        end
      
        if controller:Btn_Tool2_Press() then
          if tool_obj.ToolRightButtonPress then
            tool_obj.ToolRightButtonPress()
          end
        end
        if controller:Btn_Tool2() then
          if tool_obj.ToolRightButtonHold then
            tool_obj.ToolRightButtonHold()
          end  
        end
        if controller:Btn_Tool2_Release() then
          if tool_obj.ToolRightButtonRelease then
            tool_obj.ToolRightButtonRelease() 
          end
        end
      end
    end

  ----------- NEW VEHICLE CONTROL EXAMPLE --------------------

    -- controling the vehicle, shooting, aiming, etc.
    local vehicle_assy = pio:GetMountedToAssemblyByAT(PP.type_vehicle)
    if vehicle_assy then
      
      -- the which parameter will infrom vehicle from which mount joint the pioneer controls the vehicle (because more pioneers can be mounted to one vehicle)
      local which = ""
      if pio:GetMountedToJoint() then which = pio:GetMountedToJoint():GetName() end 
      -- ! 'GetMountedToJoint' only finds some joint to which this assembly is mounted. however this assembly can be mounted to more mount joints. i used this only knowing that pioneer can be mounted to only one thing.
      
      local vehicle_obj = GetAssemblyLuaObj(vehicle_assy)
      if vehicle_obj then 
        
        if vehicle_obj.ControlAimPoint then
          vehicle_obj.ControlAimPoint(controller:AimPoint(), which)
        end
        
        if controller:Btn_Left_Press() then 
          if vehicle_obj.ControlLeftPress then
            vehicle_obj.ControlLeftPress(which) 
          end
        end
        if controller:Btn_Left() then 
          if vehicle_obj.ControlLeftHold then
            vehicle_obj.ControlLeftHold(which) 
          end
        end
        if controller:Btn_Left_Release() then 
          if vehicle_obj.ControlLeftRelease then
            vehicle_obj.ControlLeftRelease(which) 
          end
        end
        
        if controller:Btn_Right_Press() then 
          if vehicle_obj.ControlRightPress then
            vehicle_obj.ControlRightPress(which) 
          end
        end
        if controller:Btn_Right() then 
          if vehicle_obj.ControlRightHold then
            vehicle_obj.ControlRightHold(which) 
          end
        end
        if controller:Btn_Right_Release() then 
          if vehicle_obj.ControlRightRelease then
            vehicle_obj.ControlRightRelease(which) 
          end
        end
        
        if controller:Btn_Up_Press() then 
          if vehicle_obj.ControlUpPress then
            vehicle_obj.ControlUpPress(which) 
          end
        end
        if controller:Btn_Up() then 
          if vehicle_obj.ControlUpHold then
            vehicle_obj.ControlUpHold(which) 
          end
        end
        if controller:Btn_Up_Release() then 
          if vehicle_obj.ControlUpRelease then
            vehicle_obj.ControlUpRelease(which) 
          end
        end
        
        if controller:Btn_Down_Press() then 
          if vehicle_obj.ControlDownPress then
            vehicle_obj.ControlDownPress(which) 
          end
        end
        if controller:Btn_Down() then 
          if vehicle_obj.ControlDownHold then
            vehicle_obj.ControlDownHold(which) 
          end
        end
        if controller:Btn_Down_Release() then 
          if vehicle_obj.ControlDownRelease then
            vehicle_obj.ControlDownRelease(which) 
          end
        end
        
        if controller:Btn_Tool1_Press() then 
          if vehicle_obj.ControlShoot1Press then
            vehicle_obj.ControlShoot1Press(which) 
          end
        end
        if controller:Btn_Tool1() then 
          if vehicle_obj.ControlShoot1Hold then
            vehicle_obj.ControlShoot1Hold(which) 
          end
        end
        if controller:Btn_Tool1_Release() then 
          if vehicle_obj.ControlShoot1Release then
            vehicle_obj.ControlShoot1Release(which) 
          end
        end
        
        if controller:Btn_Tool2_Press() then 
          if vehicle_obj.ControlShoot2Press then
            vehicle_obj.ControlShoot2Press(which) 
          end
        end
        if controller:Btn_Tool2() then 
          if vehicle_obj.ControlShoot2Hold then
            vehicle_obj.ControlShoot2Hold(which) 
          end
        end
        if controller:Btn_Tool2_Release() then 
          if vehicle_obj.ControlShoot2Release then
            vehicle_obj.ControlShoot2Release(which) 
          end
        end 
        
        if controller:Btn_Shield_Press() then 
          if vehicle_obj.ControlBrakePress then
            vehicle_obj.ControlBrakePress(which) 
          end
        end
        if controller:Btn_Shield() then 
          if vehicle_obj.ControlBrakeHold then
            vehicle_obj.ControlBrakeHold(which) 
          end
        end
        if controller:Btn_Shield_Release() then 
          if vehicle_obj.ControlBrakeRelease then
            vehicle_obj.ControlBrakeRelease(which) 
          end
        end
        
      end
    end


    ------------------------------------------------------------
    -- PLAYER INPUT HANDLING
    ------------------------------------------------------------

    -- keys
    local key_left_downed = controller:Btn_Left_Press()-- or controller:MoveVector().mX < -0.5
    local key_left_upped = controller:Btn_Left_Release()-- or controller:MoveVector().mX >= -0.5
    local key_right_downed = controller:Btn_Right_Press()-- or controller:MoveVector().mY > 0.5
    local key_right_upped = controller:Btn_Right_Release()-- or controller:MoveVector().mX <= 0.5
    local no_move_input = not controller:Btn_Left() and not controller:Btn_Right() and not controller:Btn_Up() and not controller:Btn_Down()
    
    -- TODO: also look at analog move input    
--    local controller:MoveVector()

    local vehicle_assy = pio:GetMountedToAssemblyByAT(PP.type_vehicle)
    if vehicle_assy then
      -- mounted to vehicle, we stop moving and flying
      pio.ride()
    else
      -- pressed to jump / fly - highest priority
--      if controller:Btn_Up() then
      if controller:MoveVector().mY > 0.83 then
        pio.fly(controller:MoveVector())
      else

          -- RELATIVE FLIGHT CONTROL MODE
        if pio.relative_flight_control then
          if controller:Btn_Up_Release() then
            pio.idle()
          end
          
          -- pressed down to go prone
          if controller:Btn_Down_Press() then
            pio.go_prone()
          end
  --[[
          -- stop sliding
          if controller:Btn_Down_Release() then
            pio.idle()
          end
  ]]--
          -- walking to the right
          if controller:Btn_Right() then
            pio.ambulate(controller:MoveVector())
          end
          if key_right_upped then
              if controller:Btn_Down() then pio.slide(controller:MoveVector().mX) else pio.idle() end
          end

          -- walking to the left
          if controller:Btn_Left() then
            pio.ambulate(controller:MoveVector())
          end
          if key_left_upped then
              if controller:Btn_Down() then pio.slide(controller:MoveVector().mX) else pio.idle() end
          end
          
        -- ABSOLUTE FLIGHT CONTROL MODE
        else
          if pio.is_flying_now() then
            -- stop flying after a while of letting go of all controls - ABSOLUTE MODE
            if controller:Btn_Up_Release() then
              pio.flight_stop_timer:Reset()
            end
            if no_move_input and pio.flight_stop_timer:Elapsed(0.25) then
              pio.idle()
            end
          else
          
            -- pressed down to go prone
            if controller:Btn_Down_Press() then
              pio.go_prone()
            end
    --[[
            -- stop sliding
            if controller:Btn_Down_Release() then
              pio.idle()
            end
    ]]--
            -- walking to the right
            if controller:Btn_Right() then
              pio.ambulate(controller:MoveVector())
            end
            if key_right_upped then
              if controller:Btn_Down() then pio.slide(controller:MoveVector().mX) else pio.idle() end
            end

            -- walking to the left
            if controller:Btn_Left() then
              pio.ambulate(controller:MoveVector())
            end
            if key_left_upped then
              if controller:Btn_Down() then pio.slide(controller:MoveVector().mX) else pio.idle() end
            end
          end
        end
      end
    end
    
    -- If not in balance and no controller input for a while, just get up and do idle
    if pio.balance_state == pio.balance.NONE and pio.can_regain_balance_now() then
      if no_move_input then
        pio.idle()
      end
    end
    
    -- tool switching
  --  if(controller:Btn_NextTool_Press()) then
  --    controller:SetNextTool() 
  --  end
  --  if(controller:Btn_PrevTool_Press()) then 
  --    controller:SetPrevTool()
  --  end
      
    --[[
    -- TEMP ONLY THE ATOMIZER FOR NOW
    if controller:CurrentToolIs("Rifle") then
      controller:SetNextTool() 
    end
    ]]--
    
    -- switch by what is in hands so far
    --local switch_to_tool = "Atomizer"
    --if pio:GetMountJointIsMounted("Tool Mount Arm BG") then
  --    switch_to_tool = "Rifle"
  --  end
    
    -- shooting
    if pio:GetAssemblyMountedToJoint("Tool Mount Arm FG") then
      -- Semi auto
      if controller:Btn_Tool1() then
        --pio.fire_weapon("Tool Rifle Rounds")
      end
      -- Full auto
      if controller:Btn_Tool2() then
        --pio.fire_weapon("Tool Rifle Freeze")
      end
    -- Atomizing/Assembling
    else
      -- Atomizing
      if controller:Btn_Tool1() then pio.atomize()  end
      if controller:Btn_Tool1_Release() then pio.idle() end
      -- Assembling
      if controller:Btn_Tool2() then pio.assemble() end
      if controller:Btn_Tool2_Release() then pio.idle() end
    end
    
    

    -- Let spot light angle by aiming
    local light = pio:GetLightByName("Spot") 
    if light ~= nil then
      local vec = controller:AimPoint()
      vec = vec - light:GetPosition()
      light:SetSpotLightAngle(vec:GetAngleRad())
    end

    

    -----------------------------------------------
    -- Mirroring Control; based on aim input relative to current positionn

    local aim_vec = pio.body_mo:UnRotateVectorForGravity(controller:AimPoint() - pio:GetAssemblyApproxPos())
    local target_is_left = aim_vec.mX < 0
    -- Detect flipping event
    -- Don't allow flipping back and forth too quickly
    if ((target_is_left and not pio:IsMirrored()) or (not target_is_left and pio:IsMirrored())) and pio.mirror_change_timer:Elapsed(0.5) then
      -- Mirror ourselves
      pio:Mirror(Yes)
      -- Measure how long since this last mirroring change
      pio.mirror_change_timer:Reset()
      
      -- If pointing at something, stop doing so on flip, or will just look silly
      if pio.jg.arm_fg.anim == pio.poseanim.point_at then
        pio.set_jg_state(pio.jg.arm_fg, pio.jgs.IDLE)
      end
      if pio.jg.arm_bg.anim == pio.poseanim.point_at then
        pio.set_jg_state(pio.jg.arm_bg, pio.jgs.IDLE)
      end
      if pio.jg.neck.anim == pio.poseanim.point_at then
        pio.set_jg_state(pio.jg.neck, pio.jgs.IDLE)
      end
      
      -- If flying, the balance angle should be adjusted to not get jerked around on a flip
      if pio.balance_state == pio.balance.FLYING then
        if target_is_left then
          pio:MOSet("Head"):SetBalanceToAngle(pio.head_mo:GetBalanceToAngle() + (math.pi / 2))
        else
          pio:MOSet("Head"):SetBalanceToAngle(pio.head_mo:GetBalanceToAngle() - (math.pi / 2))
        end
      end
    end

    ----------------------------------------------------
    -- Vehicle entry and escape

    local in_vehicle = pio:GetMountedToAssemblyByAT(PP.type_vehicle)
    if in_vehicle then
      pio.ignore_enter_vehicle_now = Yes
      pio.entering_vehicle_aid = -1
      if controller:Btn_EnterExit_Press() or in_vehicle:IsDead() then
        -- the which parameter will infrom vehicle from which mount joint the pioneer leaves the vehicle (because more pioneers can be mounted to one vehicle)
        local which = ""
        if pio:GetMountedToJoint() then which = pio:GetMountedToJoint():GetName() end 
        -- ! 'GetMountedToJoint' only finds some joint to which this assembly is mounted. however this assembly can be mounted to more mount joints. i used this only knowing that pioneer can be mounted to only one thing.
        
        local vehicle_obj = GetAssemblyLuaObj(in_vehicle)
        if vehicle_obj then 
          if vehicle_obj.OnVehiclePilotEscape then
            vehicle_obj.OnVehiclePilotEscape(which)
          end
        end
        
        -- escape the vehicle now
        pio:UnmountFromAssembly(in_vehicle)
        pio.escaping_vehicle_aid = in_vehicle:GetID()
        -- jump up
        pio.body_mo:PullTest4(pio.body_mo:RotateVectorForGravity(VectorF(0,10000)))
      else
        pio.escaping_vehicle_aid = -1
      end
    else
      if controller:Btn_EnterExit() and not pio.ignore_enter_vehicle_now then
        -- find nearest vehicle to enter
        local vehicle_to_enter = scene:GetNearestAssemblyWithFreeMountJointByAT(pio, PP.type_vehicle, 6)
        if vehicle_to_enter then
          pio.entering_vehicle_aid = vehicle_to_enter:GetID()
          pio.escaping_vehicle_aid = -1
        end
      end
      
      if controller:Btn_EnterExit_Release() then
        pio.ignore_enter_vehicle_now = No
        
        if pio.entering_vehicle_aid >= 0 then
            pio.escaping_vehicle_aid = pio.entering_vehicle_aid
            pio.entering_vehicle_aid = -1
        end
        
      end
    end
    
    
    local entering_vehicle_assy = scene:GetAssemblyByID(pio.entering_vehicle_aid)
    if entering_vehicle_assy then
      
      -- when entering the vehicle we set it to fly-out state with pioneer every frame
      pio:AddFlyOutAssembly(entering_vehicle_assy)
      
      -- get mount joint and point
      local mount_joint = entering_vehicle_assy:GetNearestAndFreeMountJoint(pio:GetAssemblyApproxPos());
      local mount_pnt = VectorF(0,0);
      if mount_joint then
        mount_pnt = mount_joint:GetWorldPositionA();
      else
        mount_pnt = entering_vehicle_assy:GetAssemblyApproxPos();
      end
      
      -- Relative vector between position to the mount point
      local mount_vec = mount_pnt - pio.body_mo:GetPosition()
      -- Un-rotate the mount point vector to gravity down so we can do proper logic on it
      mount_vec = pio.body_mo:UnRotateVectorForGravity(mount_vec)
    
      -- help the pioneer to enter the vehicle
      if(math.abs(mount_vec.mX) < 1.0) then
          pio.body_mo:PullTest4((pio.body_mo:RotateVectorForGravity(mount_vec) + pio.body_mo:RotateVectorForGravity(VectorF(0, 0.8))) * 100 - pio.body_mo:GetVelocity() * 30)
      end
      
      
      -- Run to the left if mount point is in that direction
      if mount_vec.mX < 0 then
        if not pio.is_flying_now() then
          if not pio.is_ambulating_now() then
            pio.ambulate(VectorF(-1.0, 0))
          end
          if pio.is_ambulating_now() and not pio.is_ambulating_left then
            pio.ambulate(VectorF(-1.0, 0))
          end
          if mount_vec.mX > -0.7 then
            if pio.unrotated_vel.mX < -1 or pio.unrotated_vel.mX > 1 then
              pio.idle()
            end
          end
        end
        -- Jump up to mount point if necessary
        if(mount_vec.mX > -0.7) and mount_vec.mY > 0 then 
          -- jump
          if pio.unrotated_vel.mX > -0.5 and pio.unrotated_vel.mX < 0.5 then
            --if pio.hip_joint:GetAngularSpringAngleFadeFinishedNow() then
            if not pio.wanna_jump_now and not pio.gonna_jump_now then
--              pio:FadeAngularSpringAnglesByPose("Leg", "Jump 1", pio.poseanim.jump.interval)
              pio.wanna_jump_now = true
              pio.gonna_jump_now = false
            end
          end
        end
        
      -- Run to the right if mount point is in that direction
      elseif mount_vec.mX > 0 then
        if not pio.is_flying_now() then
          if not pio.is_ambulating_now() then
            pio.ambulate(VectorF(1.0, 0))
          end
          if pio.is_ambulating_now() and pio.is_ambulating_left then
            pio.ambulate(VectorF(1.0, 0))
          end
          if mount_vec.mX < 0.7 then
            if pio.unrotated_vel.mX < -1 or pio.unrotated_vel.mX > 1 then
              pio.idle()
            end
          end 
        end
        -- Jump up to mount point if necessary
        if mount_vec.mX < 0.7 and mount_vec.mY > 0 then 
          -- jump
          if pio.unrotated_vel.mX > -1 and pio.unrotated_vel.mX < 1 then
            --if pio.hip_joint:GetAngularSpringAngleFadeFinishedNow() then
            if not pio.wanna_jump_now and not pio.gonna_jump_now then
--              pio:FadeAngularSpringAnglesByPose("Leg", "Jump 1", pio.jg.legs.anim.jump.interval)
              pio.wanna_jump_now = true
              pio.gonna_jump_now = false
            end
          end
        end
      end
      
      -- Enter the vehicle now if close enough to the mount point we are going for
      if mount_joint then 
        if pio.body_mo:GetPosition():GetLengthTo(mount_pnt)<0.8 then
          pio:MountToJoint(mount_joint, pio.body_mo, VectorF(0,0), Yes)
          
          -- the which parameter will infrom vehicle from which mount joint the pioneer leaves the vehicle (because more pioneers can be mounted to one vehicle)
          local which = mount_joint:GetName()
          local vehicle_obj = GetAssemblyLuaObj(entering_vehicle_assy)
          if vehicle_obj then 
            if vehicle_obj.OnVehiclePilotEnter then
              vehicle_obj.OnVehiclePilotEnter(which)
            end
          end
        end
      end
      
    -- Abort vehicle entry, run to the side to clear it before collisions turn on again
    else
      local escaping_vehicle = scene:GetAssemblyByID(pio.escaping_vehicle_aid)
      if escaping_vehicle then
        if not pio:GetFlyOutAssemblyFlag(escaping_vehicle) then --and pio:GetFlyOutAssembly(escaping_vehicle)  
          -- Stop escaping vehicle when not overlapping anymore
          pio.cancel_enter = true
          escaping_vehicle = nil
          pio.escaping_vehicle_aid = -1
        end
      end
      -- If we are going at speed, then just go slack and let things take their course
      if escaping_vehicle and (pio:GetApproxVelocity():GetLength() > 5.0 or escaping_vehicle:GetApproxVelocity():GetLength() > 8.0) then
        pio.slack()
      -- Run out from behind the vehicle we are not mounting anymore
      elseif escaping_vehicle then
        local exit_vec = escaping_vehicle:GetAssemblyApproxPos() - pio:GetAssemblyApproxPos()
        -- Un-rotate the exit vector for gravity so we can do logic on it
        exit_vec = pio.body_mo:UnRotateVectorForGravity(exit_vec)
        pio.cancel_enter = true
        if exit_vec.mX > 0 then
          if not pio.is_ambulating_now() then
            pio.ambulate(VectorF(-1.0, 0))
          end
          if pio.is_ambulating_now() and not pio.is_ambulating_left then
            pio.ambulate(VectorF(-1.0, 0))
          end
        end
        if exit_vec.mX < 0 then
          if not pio.is_ambulating_now() then
            pio.ambulate(VectorF(1.0, 0))
          end
          if pio.is_ambulating_now() and pio.is_ambulating_left then
            pio.ambulate(VectorF(1.0, 0))
          end
        end
      else
        if pio.cancel_enter then
          pio.cancel_enter = false
          pio.idle()
        end
      end
    end
  end


  --------------------------------------------------------
  -- Environment Sensing Functions
  --------------------------------------------------------

  function pio.get_altitude(maxCheckDistance)
    local radar_signal = pio:RayCastVector(pio:GetApproxPos(), pio.body_mo:RotateVectorForGravity(VectorF(0, -maxCheckDistance)), true, true, nil, pio)
    -- Hit something, so return the distance to it
    if radar_signal then
      return (radar_signal:GetPoint() - pio:GetApproxPos()):GetLength()
    end
    -- Did not find anyhting, so altitude is reported as what the max distance is
    return maxCheckDistance
  end
  
  function pio.raycast_leg_left(leg1)
    pio.body_mo = pio:GetMOByName("Torso")
    if not pio.body_mo then return end
    
    local leg_name = ""
    if leg1 then leg_name = "Leg FG Shin" else leg_name = "Leg BG Shin" end
    local leg = pio:GetMOByName(leg_name)
    if leg then
      local tool = pio:GetAssemblyMountedToJoint("Tool Mount Arm BG")
      if tool then tool = tool:GetMasterMO() end
      return pio:RayCastVector(leg:GetPosition(), pio.body_mo:RotateVectorForGravity(pio.raycast_legvec_left), true, true, tool, pio)
    end
    return false 
  end

  function pio.raycast_leg_right(leg1)
    pio.body_mo = pio:GetMOByName("Torso")
    if not pio.body_mo then return end
    
    local leg_name = "" 
    if leg1 then leg_name = "Leg FG Shin" else leg_name = "Leg BG Shin" end 
    local leg = pio:GetMOByName(leg_name)
    if leg then 
      local tool = pio:GetAssemblyMountedToJoint("Tool Mount Arm BG")
      if tool then tool = tool:GetMasterMO() end
      return pio:RayCastVector(leg:GetPosition(), pio.body_mo:RotateVectorForGravity(pio.raycast_legvec_right), true, true, tool, pio)
    end
    return false
  end

  ----------------------------------------------------------
  -- Pioneer State Checking Functions
  ----------------------------------------------------------

  function pio.check_fatal_damage()
    -- detect if pioneer head is even there, if not then set the pioneer dead
    if not pio.head_mo then
      pio.kill()
    -- detect if the head has been damaged, then he's also dead
    elseif pio.head_mo:GetAreaLossPerc() > 0.05 then
      pio.kill()
    end
    
    -- detect if pioneer body is even there, if not then set the pioneer dead
    if not pio.body_mo then
      pio.kill() 
    -- detect if half or more of the body is gone, then he's pretty dead
    elseif pio.body_mo:GetAreaLossPerc() > 0.5 then
      pio.kill()
    end
  end

  function pio.check_knockout_trauma()
    
--[[
    -- detect if pioneer head is even there, if not then set the pioneer dead
    if not pio.head_mo then
      pio.kill()
    -- detect if the head has been damaged, then he's also dead
    elseif pio.head_mo:GetAreaLossPerc() > 0.05 then
      pio.kill()
    end
    
    -- detect if pioneer body is even there, if not then set the pioneer dead
    if not pio.body_mo then
      pio.kill() 
    -- detect if half or more of the body is gone, then he's pretty dead
    elseif pio.body_mo:GetAreaLossPerc() > 0.5 then
      pio.kill()
    end
]]--
    
  end
  
  function pio.is_jg_ambulating_now(jointGroup)
    return jointGroup.state ~= pio.jgs.IDLE and
           (jointGroup.state == pio.jgs.WALK or
            jointGroup.state == pio.jgs.RUN or
            jointGroup.state == pio.jgs.CLIMB or
            jointGroup.state == pio.jgs.CRAWL)
  end

  function pio.is_ambulating_now()
    return pio.is_jg_ambulating_now(pio.jg.legs)
  end

  function pio.is_jumping_now()
    return pio.jg.legs.state == pio.jgs.PREJUMP or pio.jg.legs.state == pio.jgs.JUMP or pio.jg.legs.state == pio.jgs.POSTJUMP
  end

  function pio.is_riding_now()
    return pio.jg.legs.state == pio.jgs.RIDE
  end
  
  function pio.can_regain_balance_now()
    if not pio.body_mo then return false end
    if pio:GetMountedToAssemblyByAT(PP.type_vehicle) then return false end
    
    -- TODO: make this check ground speed instead of absolute speed
    if pio.body_mo:GetVelocity():GetLength() < 5.0 and math.abs(pio.body_mo:GetAngularVelocity()) < 4.5 and pio.get_altitude(3) < 2 then
      if pio.regain_balance_timer:TimeIsUp() then
        return true
      end
    else
      -- Restart timer since we don't have good conditions for getting up
      pio.regain_balance_timer:Reset()
    end
    return false
  end

  function pio.is_balancing_now()
    return pio.balance_state == pio.balance.BIPED
  end

  function pio.is_sliding_now()
    return pio.jg.legs.state == pio.jgs.SLIDE or pio.balance_state == pio.balance.GO_PRONE
  end

  function pio.is_crawling_now()
    return pio.jg.legs.state == pio.jgs.CRAWL or pio.balance_state == pio.balance.GO_PRONE
  end

  function pio.is_prone_now()
    return pio.is_sliding_now() or pio.is_crawling_now()
  end

  function pio.is_flying_now()
    return pio.jg.legs.state == pio.jgs.FLY
  end

  function pio.is_falling_now()
    return pio.balance_state == pio.balance.NONE and pio.get_altitude(3.0) > 1.0
  end

  -- Whether we need to be climbing over stuff in the direction and speed we are ambulatin'
  function pio.needs_to_climb()
    if not pio.is_ambulating_now then return false end

    if pio.is_ambulating_left then
      return pio.raycast_leg_left(pio.jg.legs.anim_num == 3)
    else
      return pio.raycast_leg_right(pio.jg.legs.anim_num == 3)
    end
    
    return false
  end


  ----------------------------------------------
  -- Limb collision detection

  function pio.legs_colliding_with_something()
    function leg_col(leg_mo)
      if leg_mo == nil or not leg_mo:IsSomeHowJointedToMO(pio.body_mo, false, false) then return false
      else return leg_mo:GetCollidedWithMOorTO() end
    end

    return leg_col(pio:GetMOByName("Leg FG Thigh"))
    or leg_col(pio:GetMOByName("Leg FG Shin"))
    or leg_col(pio:GetMOByName("Leg BG Thigh"))
    or leg_col(pio:GetMOByName("Leg BG Shin")) 
  end

  function pio.arms_colliding_with_something()    
    function arm_col(arm_mo)
      if arm_mo == nil or not arm_mo:IsSomeHowJointedToMO(pio.body_mo, false, false) then return false
      else return arm_mo:GetCollidedWithMOorTO()
      end
    end

    return arm_col(pio:GetMOByName("Arm FG Upper"))
    or arm_col(pio:GetMOByName("Arm FG Lower"))
    or arm_col(pio:GetMOByName("Arm BG Upper"))
    or arm_col(pio:GetMOByName("Arm BG Lower")) 
  end

  -- Is Pio standing/balancing with one foot on each side of his CoM?
  -- Returns movement input in x in the needed corrective direction, if any. If not, returns 0
  function pio.get_biped_balance_input()
    if pio.balance_state ~= pio.balance.BIPED then return 0 end
    
    local foot_fg_joint = pio:GetJointByName("Leg FG Foot")
    local foot_bg_joint = pio:GetJointByName("Leg BG Foot")
    -- No feet left! no balancing to do then
    if not foot_fg_joint and not foot_bg_joint then return 0 end
    
    -- Return the unrotated vector offset to the foot vector from the center of the body
    function get_foot_offset(foot_joint)
      if not foot_joint or not foot_joint:GetMOA() or not foot_joint:GetMOA():IsSomeHowJointedToMO(pio.body_mo, false, false) then return VectorF(0, 0) end
      return pio.body_mo:UnRotateVectorForGravity(foot_joint:GetWorldPositionA() - pio.body_mo:GetPosition())
    end
    
    -- Make him hop to keep balance if only one foot left
    if foot_fg_joint and not foot_bg_joint then
      local foot_offset = get_foot_offset(foot_fg_joint)
      if not pio:IsMirrored() then
        if foot_offset.mX < -0.33 then return 0.5 end
        if foot_offset.mX > 0.33 then return -0.5 end
      else
        if foot_offset.mX < -0.33 then return 0.5 end
        if foot_offset.mX > 0.33 then return -0.5 end
      end
    end
    -- Other foot, if only one leg
    if not foot_fg_joint and foot_bg_joint then
      local foot_offset = get_foot_offset(foot_bg_joint)
      if not pio:IsMirrored() then
        if foot_offset.mX < -0.33 then return 0.5 end
        if foot_offset.mX > 0.33 then return -0.5 end
      else
        if foot_offset.mX < -0.33 then return 0.5 end
        if foot_offset.mX > 0.33 then return -0.5 end
      end
    end
    
    -- We have both legs, so see if CoM is between the two
    local foot_offset_fg = get_foot_offset(foot_fg_joint)
    local foot_offset_bg = get_foot_offset(foot_bg_joint)
    -- Return corrective input if both legs are found on each side of the body
    if foot_offset_fg.mX < 0 and foot_offset_bg.mX < 0 then return 0.5 end
    if foot_offset_fg.mX > 0 and foot_offset_bg.mX > 0 then return -0.5 end
    
    -- Appears we have one foot on each side of our body, whcih is in balance
    return 0
  end


  -----------------------------------------------------------------------------------
  -- Pioneer State Changing Functions
  -----------------------------------------------------------------------------------

  -- Third param is whether to not reset the animation if this state is different
  function pio.set_jg_state(joint_group, new_jg_state, do_not_reset_anim)
    -- Mark if this state is changed for this frame, also reset the animation number if requested
    if joint_group.state ~= new_jg_state then
      joint_group.state_changed = true
      if not do_not_reset_anim then
        joint_group.anim_num = 1
      end
    end
    -- Set the current state
    joint_group.state = new_jg_state
  end

  function pio.kill()
    -- If we're already busy dying, then don't die more
    if pio.is_dying() then
      return
    end
    
    -- Turn off balance
    pio.balance_state = pio.balance.NONE
    
    -- Start counting down til we actually are fully dead
    pio.dying_timer:Reset()

    -- Everything goes into death flail
    pio.set_jg_state(pio.jg.legs, pio.jgs.DIE)
    pio.set_jg_state(pio.jg.arm_fg, pio.jgs.DIE)
    pio.set_jg_state(pio.jg.arm_bg, pio.jgs.DIE)
    pio.set_jg_state(pio.jg.neck, pio.jgs.DIE)
  end
  
  function pio.is_dying()
    return pio.jg.legs.state == pio.jgs.DIE
  end

  function pio.set_fully_dead()
    -- Everything goes slack
    pio.slack()

    -- Communicate to the engine that this thing is dead
    pio:SetDead()
  end

  -----------------------------------------------------------------------
  -- Pose Animation helpers

  -- Check and advance the pose animation to the next step, if necessary
  -- If reverse is true, the next pose selected to animate to will be the previous in order (ie playing the animation backwards)
  -- Returns whether this is non-looping and has reached the end of poses to fade to
  function pio.pose_anim_advance(joint_group, reverse)
    -- Don't do anything if this joint group just changed states
    if joint_group.state_changed then
			
			-- Do reset the hold timer if this is a single-pose non-looping state
			if joint_group.anim.count == 1 and not joint_group.anim.loop then
				joint_group.hold_timer:Reset()
				joint_group.hold_timer:SetTimeLimit(joint_group.anim.interval)
			end
			
      return false
    end
    
    -- Get any joint of this group
    local joint = pio:GetAnyJointByName(joint_group.name)
    if not joint or joint_group.anim == pio.poseanim.none then return false end
    
		-- If this is a non-looping single-pose state, then use the hold timer to see if we have held it long enough
		if joint_group.anim.count == 1 and not joint_group.anim.loop then
			-- Report that this single-pose hold has been finished
			if joint_group.hold_timer:TimeIsUp() then
				return true
			end
    -- If a looping and/or multi-pose animation, fade between the poses
		-- See if we have finished fading the springs to the current pose target
    elseif joint:GetAngularSpringAngleFadeFinishedNow() then
      -- We advance the next pose to which we should fade, depending on direction of play   
			if not reverse then
				joint_group.anim_num = joint_group.anim_num + 1
			else
				joint_group.anim_num = joint_group.anim_num - 1
			end

      -- Bounds check and loop in both directions, if looping enabled
      if joint_group.anim_num > joint_group.anim.count then
        if joint_group.anim.loop then
          -- Loop around
          joint_group.anim_num = 1
        else
          joint_group.anim_num = joint_group.anim.count
          -- Report that we have finished with the last pose of a non-looping animation now
          return true
        end
      end
      if joint_group.anim_num < 1 then
        if joint_group.anim.loop then
          -- Loop around
          joint_group.anim_num = joint_group.anim.count
        else
          joint_group.anim_num = 1
          -- Report that we have finished with the last pose of a non-looping animation now
          return true
        end
      end
    end
    -- We have not finished any non-looping animations
    return false
  end

  -- If not fading to a pose already, set all joints of a joint group to fade to its currently set anim_num in the pose animation series
  function pio.fade_to_current_pose(joint_group)
    -- Get any joint of this group
    local joint = pio:GetAnyJointByName(joint_group.name)
    if not joint or joint_group.anim == pio.poseanim.none then
      return false
    end

		-- If single-pose non-looping state, don't fade the angle spring targets, just set them immediately at the target and hold there as long as interval is
		if joint_group.anim.count == 1 and not joint_group.anim.loop then
			pio:SetAngularSpringAnglesByPose(joint_group.name, joint_group.anim.name .. " " .. joint_group.anim_num)
		-- Multi-pose animation state, so fade between the poses
		elseif not joint:GetAngularSpringAngleFadingNow() then
      -- Put together the pose animation name, and the current pose animation number in the sequence
      pio:FadeAngularSpringAnglesByPose(joint_group.name, joint_group.anim.name .. " " .. joint_group.anim_num, joint_group.anim.interval * (1 / joint_group.anim_speed))
    end
  end

  -- Aim a limb straight at a world coordinate target
  function pio.aim_joint_at_target(joint, target_pos, trim_angle)
    if joint then
      local target_vec = target_pos - joint:GetWorldPositionA()
      if pio:IsMirrored() then trim_angle = -trim_angle end
      joint:FadeAngularSpring(target_vec:GetAngleRad() - pio.body_mo:GetOrientation() + math.halfpi + trim_angle, 0)
    end
  end

  -- Set joint stiffness on all limb joints
  function pio.set_limb_joint_springs(power, friction)
    pio:SetAngularSpringPowers("Leg", power)
    pio:SetAngularSpringPowers("Arm", power)
    pio:SetAngularSpringPowers("Neck", power)
    pio:SetAngularFrictions("Leg", friction)
    pio:SetAngularFrictions("Arm", friction)
    pio:SetAngularFrictions("Neck", friction)
  end

  ------------------------------------------------
  -- IDLE

  function pio.idle()
    pio.body_mo = pio:GetMOByName("Torso")
  --  pio.jg.legs.anim_num = 1
    pio.ambulation_input = VectorF(0, 0)
    
    -- Start falling flailing if we are too high up
    if pio.get_altitude(5) > 4 then
      pio.fall()
      return
    end

    -- Get up on our feet if we are not yet
    if pio.balance_state ~= pio.balance.BIPED then
      pio.gain_balance()
      return
    end
--[[ TODO: Make this work
    -- If our legs are not on each side of our CoM, we need to take some corrective steps to get our feet under our Pio
    local balance_input = pio.get_biped_balance_input()
    if math.abs(balance_input) > 0 then
      pio.ambulate(VectorF(balance_input, 0))
      return
    end
]]--
    -- Turn off jetpack
    local jet = pio:GetPSByName("Jetpack")
    if jet then jet:SwitchOFFParticleEmitting()  end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_stiff * 0.8, pio.angular_spring_friction)
    
    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, pio.jgs.IDLE)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.IDLE)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.IDLE)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.IDLE)
--    end

    if not pio.body_mo then return end
		
    -- If we are going fast enough, do the skidding animation
  -- TODO: make this look at ground speed instead of absolute speed, so it'll behave right when on a moving MO platform
    if math.abs(pio.body_mo:UnRotateVectorForGravity(pio.body_mo:GetVelocity()).mX) > 6 then
      pio.skid()
    end
  end


  ------------------------------------------------
  -- SLACK

  function pio.slack()
    pio.body_mo = pio:GetMOByName("Torso")
  --  pio.jg.legs.anim_num = 1
    pio.balance_state = pio.balance.NONE
    pio.ambulation_input = VectorF(0, 0)
    
    -- See if the conditions for regaining balance are there
    if pio.can_regain_balance_now() then
      pio.gain_balance()
      return
    end
    
    -- Turn off jetpack
    local jet = pio:GetPSByName("Jetpack")
    if jet then  jet:SwitchOFFParticleEmitting()  end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(100, 10)
    pio:SetAngularFrictions("Neck", pio.angular_spring_friction)
    
    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, pio.jgs.SLACK)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.SLACK)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.SLACK)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.SLACK)
--    end
  end


  ------------------------------------------------
  -- RIDE

  function pio.ride()
    pio.body_mo = pio:GetMOByName("Torso")
  --  pio.jg.legs.anim_num = 1
    pio.balance_state = pio.balance.NONE
    pio.ambulation_input = VectorF(0, 0)
    
    -- Turn off jetpack
    local jet = pio:GetPSByName("Jetpack")
    if jet then  jet:SwitchOFFParticleEmitting()  end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(100, 10)
    pio:SetAngularSpringPowers("Neck", pio.angular_spring_power_loose)
    pio:SetAngularSpringPowers("Arm", pio.angular_spring_power_stiff)
    pio:SetAngularSpringPowers("Leg", pio.angular_spring_power_stiff)
    pio:SetAngularFrictions("Neck", pio.angular_spring_friction)
        
    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, pio.jgs.RIDE)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.RIDE)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.RIDE)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.RIDE)
--    end
  end


  ------------------------------------------------
  -- GAIN BALANCE

  function pio.gain_balance()
    pio.body_mo = pio:GetMOByName("Torso")
    
    -- If already in balance, then just idle
    if pio.balance_state == pio.balance.BIPED then
      pio.idle()
      return
    end
    
    -- See if the conditions for regaining balance are there
    if not pio.can_regain_balance_now() then
      -- If not, then just relax on ground, to come to a halt so we can get up on our feet
      pio.slack()
      return
    end
    
    -- Start balancing
    pio.balance_state = pio.balance.GET_UP
    pio.ambulation_input = VectorF(0, 0)
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_stiff, pio.angular_spring_friction)
    
    -- Get the angle of the body
    -- This gets a value between 0 and 2PI, so [0 - PI) is rotated to the left of standing, and [PI - 2PI) is right of standing
    local bodyAngle = pio.body_mo:RotateAngleForGravity(pio.body_mo:GetOrientation())
    
    -- Determine which animation we should use to push ourselves up, depending on facing ground or not
    local direction = pio.jgs.GET_UP_BACK
    if (pio:IsMirrored() and bodyAngle < math.pi) or (not pio:IsMirrored() and bodyAngle > math.pi) then
      direction = pio.jgs.GET_UP_FRONT
    end
    
    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, direction)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, direction)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, direction)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, direction)
--    end
  end


  ------------------------------------------------
  -- GO PRONE

  function pio.go_prone()
    pio.body_mo = pio:GetMOByName("Torso")
    pio.balance_state = pio.balance.NONE
    pio.ambulation_input = VectorF(0, 0)
    
    if pio.jg.legs.state == pio.jgs.SLIDE then
      return
    end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_stiff, pio.angular_spring_friction)
    
    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, pio.jgs.GO_PRONE)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.GO_PRONE)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.GO_PRONE)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.GO_PRONE)
--    end
  end


  --------------------------------------------------------
  -- AMBULATE (walk, run, climb, crawl)

  -- Input is signed float scalar, -1.0 (max run to the left) to 1.0 (max run to the right)
  function pio.ambulate(input)
    -- If already moving do not reset animations
    local do_not_reset_animations = pio.is_ambulating_now()

    -- Start falling flailing if we are too high up
    if pio.get_altitude(5) > 4 then
      pio.fall()
      return
    end

    -- If flying or falling, don't ambulate
    if pio.is_flying_now() or pio.is_falling_now() then
      return
    end

    -- Update the non-joint group states 
    pio.ambulation_input = input
    pio.is_ambulating_left = pio.ambulation_input.mX < 0
    
    -- IDLE
    -- If we are not really getting any input, then don't ambulate, just go back to idle
    if math.abs(pio.ambulation_input.mX) < 0.2 then
      -- Player is pressing down, stay prone
      if pio.ambulation_input.mY < -0.5 then
        pio.go_prone()
        return
      -- Just stand up
      else
        pio.idle()
        return
      end
    end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_stiff, pio.angular_spring_friction)

    -- CRAWLING
    if pio.ambulation_input.mY < -0.5 then
      -- No balance state
      pio.balance_state = pio.balance.PRONE
      
      -- Set the legs' state
      pio.set_jg_state(pio.jg.legs, pio.jgs.CRAWL, do_not_reset_animations)
      -- Set the arms' and neck state, if they're not doing anything else
      -- FG
 --     if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
        pio.set_jg_state(pio.jg.arm_fg, pio.jgs.CRAWL, do_not_reset_animations)
 --     end
      -- BG
 --     if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
        pio.set_jg_state(pio.jg.arm_bg, pio.jgs.CRAWL, do_not_reset_animations)
 --     end
      -- Neck
 --     if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
        pio.set_jg_state(pio.jg.neck, pio.jgs.CRAWL, do_not_reset_animations)
 --     end
  
    -- CLIMBING
    -- Detect if we should be climbing, or if we've started climbing and we should keep doing it for a bit o avoid spazzing between running and climbing
    elseif pio.needs_to_climb() or not pio.climbing_timer:TimeIsUp() then
      -- No balance state
      pio.balance_state = pio.balance.PRONE
      
      -- Start timing how long we've been climbing
      if pio.jg.legs.state ~= pio.jgs.CLIMB then
        pio.climbing_timer:Reset()
      end
      
      -- Set the legs' state
      pio.set_jg_state(pio.jg.legs, pio.jgs.CLIMB, do_not_reset_animations)
      -- Set the arms' and neck state, if they're not doing anything else
      -- FG
      if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
        pio.set_jg_state(pio.jg.arm_fg, pio.jgs.CLIMB, do_not_reset_animations)
      end
      -- BG
      if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
        pio.set_jg_state(pio.jg.arm_bg, pio.jgs.CLIMB, do_not_reset_animations)
      end
      -- Neck
      if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
        pio.set_jg_state(pio.jg.neck, pio.jgs.CLIMB, do_not_reset_animations)
      end
      
    -- Not climbing; on flat enough surface
    else
    
      -- Biped balance state
      pio.balance_state = pio.balance.BIPED
      
      -- RUNNING
      -- Determine whether we should be walking or running with this input
      -- Also don't allow running if both legs are not present
      if math.abs(pio.ambulation_input.mX) > 0.75 and pio:GetMOByName("Leg FG Shin") and pio:GetMOByName("Leg BG Shin") then
        -- Set the legs' state
        pio.set_jg_state(pio.jg.legs, pio.jgs.RUN, do_not_reset_animations)
        -- Set the arms' and neck state, if they're not doing anything else
        -- FG
        if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
          pio.set_jg_state(pio.jg.arm_fg, pio.jgs.RUN, do_not_reset_animations)
        end
        -- BG
        if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
          pio.set_jg_state(pio.jg.arm_bg, pio.jgs.RUN, do_not_reset_animations)
        end
        -- Neck
        if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
          pio.set_jg_state(pio.jg.neck, pio.jgs.RUN, do_not_reset_animations)
        end
      
      -- WALKING
      else
        -- Set the legs' state
        pio.set_jg_state(pio.jg.legs, pio.jgs.WALK, do_not_reset_animations)
        -- Set the arms' and neck state, if they're not doing anything else
        -- FG
        if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
          pio.set_jg_state(pio.jg.arm_fg, pio.jgs.WALK, do_not_reset_animations)
        end
        -- BG
        if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
          pio.set_jg_state(pio.jg.arm_bg, pio.jgs.WALK, do_not_reset_animations)
        end    
        -- Neck
        if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
          pio.set_jg_state(pio.jg.neck, pio.jgs.WALK, do_not_reset_animations)
        end   
      end
    end
  end
  
  
  ------------------------------------------------
  -- JUMP

  function pio.jump()
    pio.body_mo = pio:GetMOByName("Torso")
    -- switch off jet particles
--    local jet = pio:GetPSByName("Jetpack")
--    if jet then jet:SwitchOFFParticleEmitting() end
    
    -- Only proceed with jump if we're close to the ground
    if pio.get_altitude(4) > 2 then
      pio.idle()
      return
    end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_stiff, pio.angular_spring_friction)
    
    -- Set balance state to bipedal - TODO: Or special jumping one?
    pio.balance_state = pio.balance.BIPED

    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, pio.jgs.PREJUMP)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.PREJUMP)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.PREJUMP)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.PREJUMP)
--    end
  end
  
  
  
  ------------------------------------------------
  -- FALL

  function pio.fall()
    pio.body_mo = pio:GetMOByName("Torso")
    pio.ambulation_input = VectorF(0, 0)
    pio.jetpack_input = 0
    -- switch off jet particles
    local jet = pio:GetPSByName("Jetpack")
    if jet then jet:SwitchOFFParticleEmitting() end
    
    -- If we are not very high over ground, stop flailing and just go slack for impact
    if pio.get_altitude(4) < 3 then
      pio.slack()
      return
    end
    
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_loose, 10)
    
    -- Set balance state to none
    pio.balance_state = pio.balance.NONE

    -- Set the joint groups to idle if and only if they were ambulating before
    pio.set_jg_state(pio.jg.legs, pio.jgs.FALL)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.FALL)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.FALL)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.FALL)
--    end
  end
  
  
  ------------------------------------------------
  -- FLY

  function pio.fly()
    pio.body_mo = pio:GetMOByName("Torso")
    pio.ambulation_input = VectorF(0, 0)
    pio.jetpack_input = input
    
    -- Make sure we have a jetpack and Aurium charge to fly with, or go back to idle
    local jet = pio:GetPSByName("Jetpack")
--[[ Lame; do proper jump animation instead
		-- If have no gas, still give a puff to give a usable jump, and to show we are out of gas
		if jet and pio.aurium_charge == 0 and pio.total_flight_timer:GetTime() > 0.5 then
      pio.total_flight_timer:Reset()
			jet:EmitBoom()
		end
]]--
		-- If jet is out of commission or fuel, do a regular jump instead
    if not jet or pio.aurium_charge <= 0 then
			if not pio.is_jumping_now() then
				pio.jump()
			end
--      pio.idle()
			-- Still mark since last Aurium use - it was attempted
			pio.aurium_charge_change_timer:Reset()
      return
    end
    
    -- Set balance state to flyin' 
    pio.balance_state = pio.balance.FLYING
    
    -- Start flight if we have enough Aurium charge to do so
    if not pio.is_flying_now() and pio.aurium_charge > 50 then
      -- Boom first when not on - and only if aurium charge is at max
--			if pio.aurium_charge >= pio.aurium_capacity then - nope, makes it feel like thrust is unpredicable
			-- Have a simple delay between booms so can't just peppra knappen
			if pio.total_flight_timer:GetTime() > 0.5 then
				jet:EmitBoom()
			end
      -- Start timing how long we've been flying
      pio.total_flight_timer:Reset()
-- TEMP DEBUG REMOVE
      if not pio:GetController():Btn_Up_Press() then
        local bajs = 9
      end
--[[
      -- Set the initial balancing angle to be horizontal if we are in a prone position
      if pio.is_prone_now() then
        if pio:IsMirrored() then
          pio:JointSet("Jet Nozzle"):SetAngularSpringAngle(pio.body_mo:RotateAngleForGravity(-math.pi / 2))
        else
          pio:JointSet("Jet Nozzle"):SetAngularSpringAngle(pio.body_mo:RotateAngleForGravity(math.pi / 2))
        end
      end
]]--
      -- Set the initial balancing angle to be whatever the head is at currently
--      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:GetOrientation() - math.halfpi)
      pio:MOSet("Head"):SetBalanceToAngle(-pio.head_mo:GetOrientation())
    end

    -- Switch on jet particles, only after a short period so initial boom doesn't interfere with it
    if pio.total_flight_timer:Elapsed(0.05) then
      jet:SwitchONParticleEmitting()
    end

    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_loose, pio.angular_spring_friction)
    -- Neck stiff since steering/stabilizing with blaancer on head
    pio:SetAngularSpringPowers("Neck", pio.angular_spring_power_stiff)
    pio:SetAngularFrictions("", pio.angular_spring_friction)
    
    -- Set the joint groups to fly
    pio.set_jg_state(pio.jg.legs, pio.jgs.FLY)
    
--    if pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.FLY)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.arm_bg) then 
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.FLY)
--    end
--    if pio.is_jg_ambulating_now(pio.jg.neck) then 
      pio.set_jg_state(pio.jg.neck, pio.jgs.FLY)
--    end
  end

  --------------------------------------------------------
  -- SLIDE (slide on a cushion of aurium force field)

  -- Input is signed float scalar, -1.0 (max run to the left) to 1.0 (max run to the right)
  function pio.slide(input)
    -- Update the non-joint group states
    pio.sliding_input = input
    pio.balance_state = pio.balance.NONE
    
    -- Start falling flailing if we are too high up
    if pio.get_altitude(5) > 4 then
      pio.fall()
      return
    end
    
--[[ Get back on feet when down is released instead
    -- If we are not really getting any input, then don't slide, just get back on our feet
    if math.abs(input) < 0.1 then
      pio.gain_balance()
    end
]]--
    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_loose, pio.angular_spring_friction)
    
    -- Set the legs' state
    pio.set_jg_state(pio.jg.legs, pio.jgs.SLIDE)
    -- Set the arms' and neck state, if they're not doing anything else
    -- FG
--    if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.SLIDE)
--    end
    -- BG
--    if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.SLIDE)
--    end
    -- Neck
--    if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
      pio.set_jg_state(pio.jg.neck, pio.jgs.SLIDE)
--    end
  end


  ------------------------------------------------------
  -- SKID

  function pio.skid()
    pio.balance_state = pio.balance.BIPED
    
    pio.body_mo = pio:GetMOByName("Torso")

    -- Set up te joint springs
    pio.set_limb_joint_springs(pio.angular_spring_power_stiffer, pio.angular_spring_friction)
    
    -- Legs
    pio.set_jg_state(pio.jg.legs, pio.jgs.SKID)
    -- Set the arms' and neck state, if they're not doing anything else
    -- FG
--    if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.SKID)
--    end
    -- BG
--    if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.SKID)
--    end
    -- Neck
--    if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
      pio.set_jg_state(pio.jg.neck, pio.jgs.SKID)
--    end
    
    if not pio.body_mo then return end
    -- Give the guy a last lil push upward so he can tuck his feet under to stop
    --pio.body_mo:PullTest4(pio.body_mo:RotateVectorForGravity(VectorF(0, 2000)))
  end


  ------------------------------------------------------
  -- ATOMIZE

  function pio.atomize()    
    pio.body_mo = pio:GetMOByName("Torso")
    
    -- Does not involve Legs
--    pio.set_jg_state(pio.jg.legs, pio.jgs.SKID)
    -- Set only the BG arm and neck's state, nothing else is involved in atomizing
    -- FG
--    if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
--      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.SKID)
--    end
    -- BG
--    if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.ATOMIZE)
--    end
    -- Neck
--    if pio.jg.neck.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.neck) then
      pio.set_jg_state(pio.jg.neck, pio.jgs.ATOMIZE)
--    end
    
  end


  ------------------------------------------------------
  -- ASSEMBLE

  function pio.assemble()    
    pio.body_mo = pio:GetMOByName("Torso")
    
    -- Does not involve Legs
--    pio.set_jg_state(pio.jg.legs, pio.jgs.SKID)
    -- Set only the BG arm and neck's state, nothing else is involved in atomizing
    -- FG
--    if pio.jg.arm_fg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_fg) then
        pio.set_jg_state(pio.jg.arm_fg, pio.jgs.ASSEMBLE)
--    end
    -- BG
--    if pio.jg.arm_bg.state == pio.jgs.IDLE or pio.is_jg_ambulating_now(pio.jg.arm_bg) then
--      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.POINT_AT)
--    end
    -- Neck
    if pio.jg.neck.state == pio.jgs.IDLE then
      pio.set_jg_state(pio.jg.neck, pio.jgs.ASSEMBLE)
    end
    
  end


  ---------------------------------------------------------
  -- Weapon tool firing

  function pio.fire_weapon(weapon_name)
    local bullets = pio:GetPSByName(weapon_name)
    if bullets and pio.last_fire_timer:TimeIsUp() then
      local vec = pio:GetController():AimPoint()
      vec = vec - bullets:GetPosition()
      vec:NormalizeTo(80)
      bullets:SetStartMoveVector(vec)
      bullets:EmitOne() -- emit one particle from the bullets particle system
      pio.last_fire_timer:Reset() -- time since last shot
    end
  end


  --------------------------------------------------------
  -- Debug drawing
   
  function pio.draw_raycasts()
  --[[
    local rayvec = VectorF(0,0)
    if pio:IsMirrored() then rayvec = pio.raycast_legvec_left else rayvec = pio.raycast_legvec_right end
    local color
    if pio.jg.legs.anim == pio.poseanim.run then color = ColorRGBA(255,255,255,255) else color = ColorRGBA(255,55,55,255) end
    local leg = pio:GetMOByName("Leg FG Shin") 
    if leg then
      local leg_pos = leg:GetPosition()
      leg:DrawWireLine(leg_pos, leg_pos + rayvec, color )
    end 
    local leg = pio:GetMOByName("Leg BG Shin")
    if leg then
      local leg_pos = leg:GetPosition()
      leg:DrawWireLine(leg_pos, leg_pos + rayvec, color )
    end 
  ]]
    


    -- Draw orientation and balance target vectors so we can see what's going on
    orientationVec = VectorF(0, 1.0)
    accelerationVec = VectorF(0, -1.0)
    balanceTargetVec = VectorF(0, 1.0)

    local body = pio:GetMOByName("Torso")
    if body then
      orientationVec:RotateLeft(body:GetOrientation())
      -- Get the gravity acceleration that is acting on the body right now
      body:RotateVectorForGravity(accelerationVec)
      -- Make the balance target's length correspond with the balance spring strength
      balanceTargetVec.mY = body:GetBalancerPower() / 3000;
      -- Need to rotate this the opposite way?
      balanceTargetVec:RotateRight(body:GetBalanceToAngle())
      local body_pos = body:GetPosition()
      body:DrawWireLine(body_pos, body_pos + orientationVec, ColorRGBA(255, 255, 255, 255))
      body:DrawWireLine(body_pos, body_pos + accelerationVec, ColorRGBA(255, 0, 0, 255))
      -- Only draw balance if it's active
      if body:GetBalancer() then
        body:DrawWireLine(body_pos, body_pos + balanceTargetVec, ColorRGBA(0, 255, 0, 255))
      end
    end
  end
 
  -- Build UI resource bars etc
  pioneer_ui.build(pio)
end



-------------------------------------------------------------------
-------------------------------------------------------------------
-- UPDATE
-------------------------------------------------------------------
-------------------------------------------------------------------



-- Pioneer standardized Update. This is called at the beginning of each frame update in its own script
function pioneer_std.update(pio)
  if not pio then return end
  local ctrl = pio:GetController()
  if not ctrl then return end
	

	
  --local my_table = scene:GetAlerts()
  --scene:WriteToScreen(tostring(my_table[0]['a22']), VectorF(300,300), No)
  --scene:WriteToScreen(tostring(my_table[0][1]), VectorF(300,280), No)
  --scene:WriteToScreen(tostring(my_table[1][0]), VectorF(300,260), No)
  
  --for k, v in pairs(my_table) do
  --scene:WriteToScreen(tostring(v), VectorF(300,300), No)
  --end
  --if my_table[0][0]==22 then return end
  --if my_table[0]==22 then return end
  


  --------------------------------------------------
  -- Get Frequently Used Pointers
  -- here we get fresh pointers to head, body and one leg MOs
  -- we always have to get fresh pointers to pio and all its objects, because pio and its objects can be brokem
  -- we assume that pointers to any pio object can change in any play-frame 
  pio.head_mo = pio:GetMOByName("Head")
  pio.body_mo = pio:GetMOByName("Torso")
  pio.hip_joint = pio:GetJointByName("Leg FG Hip")
  if pio.hip_joint == nil then pio.hip_joint = pio:GetJointByName("Leg BG Hip") end
  -- Also un-rotate the body velocity so we can do simple logic on it
  if pio.body_mo then
    pio.unrotated_vel = pio.body_mo:UnRotateVectorForGravity(pio.body_mo:GetVelocity())
  end


  -----------------------------------------------------
  -- Reset animation speed modifications
  pio.jg.legs.anim_speed = 1.0
  pio.jg.arm_fg.anim_speed = 1.0
  pio.jg.arm_bg.anim_speed = 1.0
  pio.jg.neck.anim_speed = 1.0
  

  ----------------------------------------------------------
  -- Debug Drawing

  -- draw raycasts
  -- pio.draw_raycasts()

  -- draw vector to up and rotate it by gravity
  --local vec = VectorF(0,2)
  --vec = pio.body_mo:RotateVectorForGravity(vec)
  --pio.body_mo:DrawWireVector(pio.body_mo:GetPosition(), vec, ColorRGBA(255,255,255,255))


  --------------------------------------------------------------
  -- Check if this is still alive and well 
  
  -- Do we have fatal damage and should start dying?
  pio.check_fatal_damage()
  
  -- Check if we are dying; proceed to become fully dead
  if pio.is_dying() and pio.dying_timer:TimeIsUp() and pio:IsAlive() then
    pio.set_fully_dead()
  end
--[[
  -- If no body, then we ded
  if not (pio.body_mo and pio:IsAlive()) then
    return
  end
]]--
  -- Check for controller
  local controller = pio:GetController()
  -- If we don't have a controller, just stand there -- improve; code below needs to run too even when no controller
  if not controller then
    if pio.body_mo then
--      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(pio.balance_lean))
--      pio:MOSet("Torso"):SetBalancerPower(pio.balance_power)
      pio.balance_state = pio.balance.NONE
    end
  -- Handle all controller input
  elseif pio:IsDead() then
    -- Do nothing?
  else
    if not controller:GetMenuVisible() and not controller:GetIsRemoteControlling() then
      -- handle input if not remote controlling something else and not in menu now
      pio.handle_input(controller)
    else
      --stop ambulation and regain balance if needed
      if pio.ambulation_input ~= VectorF() or pio.jetpack_input ~= 0 then
        pio.jetpack_input = 0
        pio.idle()
      elseif pio.balance_state == pio.balance.NONE and pio.can_regain_balance_now() then
        pio.idle()
      end
    end
  end

  -----------------------------------------------------------
  -- Update animations and physics based on current states
  -----------------------------------------------------------

  ---------------------------------
  -- LEGS JOINT GROUP STATE

  -- IDLE
  if pio.jg.legs.state == pio.jgs.IDLE then
  
    -- Fall over if we can't be in balance due to motion
--    if not pio.can_regain_balance_now() then
    if pio.get_altitude(3) > 2 then
      pio.fall()
    else
--[[ TODO: Make this work
      -- If our legs are not on each side of our CoM, we need to take some corrective steps to get our feet under our Pio
      local balance_input = pio.get_biped_balance_input()
      if math.abs(balance_input) > 0 then
        pio.regain_balance(VectorF(balance_input, 0))
      -- Ok, feet planted on steady ground.. just stand there
      else
]]--
        pio.jg.legs.anim = pio.poseanim.idle
        pio.pose_anim_advance(pio.jg.legs, false)    
        pio.balance_state = pio.balance.BIPED
        pio.balance_lean = 0
--      end
    end
  -- DIE
  elseif pio.jg.legs.state == pio.jgs.DIE then
    pio.jg.legs.anim = pio.poseanim.die
    pio.pose_anim_advance(pio.jg.legs, false)
    pio.set_limb_joint_springs(pio.angular_spring_power_loose, pio.angular_spring_friction)
    
    pio.balance_state = pio.balance.NONE
    pio.balance_lean = 0
  -- SLACK
  elseif pio.jg.legs.state == pio.jgs.SLACK then
    pio.jg.legs.anim = pio.poseanim.slack
    pio.balance_state = pio.balance.NONE
  -- BRACE
  elseif pio.jg.legs.state == pio.jgs.BRACE then
    pio.jg.legs.anim = pio.poseanim.brace
    -- Animate the animation backwards if facing opposite way of last direction of travel
    if pio.pose_anim_advance(pio.jg.legs, (pio:IsMirrored() and not pio.is_ambulating_left) or (not pio:IsMirrored() and pio.is_ambulating_left)) then
      pio.idle()
    end
    pio.balance_lean = 0.15
  -- RIDE
  elseif pio.jg.legs.state == pio.jgs.RIDE then
    pio.jg.legs.anim = pio.poseanim.ride
    pio.pose_anim_advance(pio.jg.legs, false)
    pio.balance_state = pio.balance.NONE
  -- STEP
  elseif pio.jg.legs.state == pio.jgs.STEP then
    pio.jg.legs.anim = pio.poseanim.step
    -- Animate the animation backwards if facing opposite way of last direction of travel
    if pio.pose_anim_advance(pio.jg.legs, (pio:IsMirrored() and not pio.is_ambulating_left) or (not pio:IsMirrored() and pio.is_ambulating_left)) then
      pio.idle()
    end
    pio.balance_state = pio.balance.BIPED
    pio.balance_lean = 0.1
  -- WALK
  elseif pio.jg.legs.state == pio.jgs.WALK then
    pio.jg.legs.anim = pio.poseanim.walk
    -- Animate the animation backwards if facing opposite way of last direction of travel
    if pio.pose_anim_advance(pio.jg.legs, (pio:IsMirrored() and not pio.is_ambulating_left) or (not pio:IsMirrored() and pio.is_ambulating_left)) then
      pio.idle()
    end
    -- forward lean when walking
    pio.balance_state = pio.balance.BIPED
    pio.balance_lean = 0.1
  -- RUN
  elseif pio.jg.legs.state == pio.jgs.RUN then
    pio.jg.legs.anim = pio.poseanim.run
    -- Set speed when running vs going reverse
    local reverse = (pio:IsMirrored() and not pio.is_ambulating_left) or (not pio:IsMirrored() and pio.is_ambulating_left)
    if reverse then
      pio.jg.legs.anim_speed = 0.5
    else
      pio.jg.legs.anim_speed = 1.0
    end

			-- Pause run animation if legs are not going to make contact with anything below
		local wait_leg_run = false
		local leg_mo = nil
		-- Pick which leg is relevant in the run cycle
		if pio.jg.legs.anim_num == 2 then
			leg_mo = pio:GetMOByName("Leg FG Shin")
		end
		if pio.jg.legs.anim_num == 5 then
			leg_mo = pio:GetMOByName("Leg BG Shin")
		end
		
		-- If in the proper place in the run cycle (one leg full forward), pause if sensor shows that we're not in touch with the ground
		if leg_mo then
			local leg_radar_len = 0.4
			local pos = leg_mo:GetPosition()
			local vec = VectorF(leg_radar_len,0)
			local col = VectorF(0,0)
			local sig = false
			local i
			for i=0, 7, 1 do
				sig = sig or scene:RayCastVector(pos, vec, col, pio)
				vec:RotateLeft(0.7854)
			end
			wait_leg_run = not sig
		end
		
		-- If we're not pausing mid-stride to get closer to ground, go ahead and move them legs
		if not wait_leg_run then			
			-- Animate the animation backwards if facing opposite way of last direction of travel
			if pio.pose_anim_advance(pio.jg.legs, reverse) then
				pio.idle()
			end
		end
		
    -- Biped balanced
    pio.balance_state = pio.balance.BIPED
    -- forward lean when running
    pio.balance_lean = 0.2
  -- CLIMB
  elseif pio.jg.legs.state == pio.jgs.CLIMB then
    pio.jg.legs.anim = pio.poseanim.climb
    -- Animate the animation backwards if facing opposite way of last direction of travel
    if pio.pose_anim_advance(pio.jg.legs, (pio:IsMirrored() and not pio.is_ambulating_left) or (not pio:IsMirrored() and pio.is_ambulating_left)) then
      pio.idle()
    end
    -- Determine the appropriate lean based on ray casts telling us about angle of normal of MO/TO we're on
  -- TODO
    pio.balance_state = pio.balance.BIPED
    pio.balance_lean = 0.6
  -- CRAWL
  elseif pio.jg.legs.state == pio.jgs.CRAWL then
    pio.jg.legs.anim = pio.poseanim.crawl
    -- Animate the animation backwards if facing opposite way of last direction of travel
    if pio.pose_anim_advance(pio.jg.legs, (pio:IsMirrored() and not pio.is_ambulating_left) or (not pio:IsMirrored() and pio.is_ambulating_left)) then
      pio.idle()
    end
  -- GET UP BACK
  elseif pio.jg.legs.state == pio.jgs.GET_UP_BACK then
    pio.jg.legs.anim = pio.poseanim.get_up_back
    if pio.pose_anim_advance(pio.jg.legs, false) then
      -- We are now in balance on our feet
      pio.balance_state = pio.balance.BIPED
      pio.idle()
    end
  -- GET UP FRONT
  elseif pio.jg.legs.state == pio.jgs.GET_UP_FRONT then
    pio.jg.legs.anim = pio.poseanim.get_up_front
    if pio.pose_anim_advance(pio.jg.legs, false) then
      -- We are now in balance on our feet
      pio.balance_state = pio.balance.BIPED
      pio.idle()
    end
  -- GO PRONE
  elseif pio.jg.legs.state == pio.jgs.GO_PRONE then
    -- Balancers pull down body to horizontal
    pio.balance_state = pio.balance.GO_PRONE

    pio.jg.legs.anim = pio.poseanim.go_prone
    if pio.pose_anim_advance(pio.jg.legs, false) then
      -- After done going prone, start sliding state
      pio.slide(0)
    end
  -- TEETER
  elseif pio.jg.legs.state == pio.jgs.TEETER then
    pio.jg.legs.anim = pio.poseanim.teeter
    if pio.pose_anim_advance(pio.jg.legs, false) then
      pio.idle()
    end
    pio.balance_state = pio.balance.NONE
  -- PREJUMP
  elseif pio.jg.legs.state == pio.jgs.PREJUMP then
    pio.jg.legs.anim = pio.poseanim.prejump
		-- When done with prejump, go to actual jump
    if pio.pose_anim_advance(pio.jg.legs, false) then
			pio.set_jg_state(pio.jg.legs, pio.jgs.JUMP)
		end
    pio.balance_state = pio.balance.BIPED
  -- JUMP
  elseif pio.jg.legs.state == pio.jgs.JUMP then
    pio.jg.legs.anim = pio.poseanim.jump
		-- When done with jump, go back to idle
    if pio.pose_anim_advance(pio.jg.legs, false) then
			pio.set_jg_state(pio.jg.legs, pio.jgs.POSTJUMP)
		end
    pio.balance_state = pio.balance.BIPED
  -- POSTJUMP
  elseif pio.jg.legs.state == pio.jgs.POSTJUMP then
    pio.jg.legs.anim = pio.poseanim.postjump
		-- When done with jump, go back to idle
    if pio.pose_anim_advance(pio.jg.legs, false) then
			pio.idle()
		end
    pio.balance_state = pio.balance.BIPED
  -- FALL
  elseif pio.jg.legs.state == pio.jgs.FALL then
    pio.jg.legs.anim = pio.poseanim.fall
    pio.pose_anim_advance(pio.jg.legs, false)
    -- Stop falling flail when close to the ground
    if pio.get_altitude(2) < 1 then
      pio.idle()
    end
    pio.balance_state = pio.balance.NONE
  -- FLY
  elseif pio.jg.legs.state == pio.jgs.FLY then
    pio.jg.legs.anim = pio.poseanim.fly
--    if pio.pose_anim_advance(pio.jg.legs, false) then
--      pio.idle()
--    end

		-- Note that we are using Aurium
		pio.no_aurium_use_timer:Reset()

    pio.balance_state = pio.balance.FLYING
  -- SKID
  elseif pio.jg.legs.state == pio.jgs.SKID then
    pio.jg.legs.anim = pio.poseanim.skid
    if pio.pose_anim_advance(pio.jg.legs, false) then
      pio.set_jg_state(pio.jg.legs, pio.jgs.IDLE)
    end
    if pio.is_ambulating_left then
      -- Counter lean for a lil bit so we can come screetching to a halt
      pio.balance_lean = 0.8
    else
      pio.balance_lean = -0.8
    end
    pio.balance_state = pio.balance.BIPED
--    pio.balance_power = 6000
  -- SLIDE
  elseif pio.jg.legs.state == pio.jgs.SLIDE then
    pio.jg.legs.anim = pio.poseanim.slide
    if pio.pose_anim_advance(pio.jg.legs, false) then
      pio.idle()
    end
    pio.balance_state = pio.balance.NONE
  -- GRAB TOOL
  elseif pio.jg.legs.state == pio.jgs.GRAB_TOOL then
    pio.jg.legs.anim = pio.poseanim.none
  -- DRAW TOOL
  elseif pio.jg.legs.state == pio.jgs.DRAW_TOOL then
    pio.jg.legs.anim = pio.poseanim.none
  -- AIM TOOL AT
  elseif pio.jg.legs.state == pio.jgs.AIM_TOOL_AT then
    pio.jg.legs.anim = pio.poseanim.none
  -- POINT AT
  elseif pio.jg.legs.state == pio.jgs.POINT_AT then
    pio.jg.legs.anim = pio.poseanim.none
  end

  ---------------------------------
  -- ARM FG JOINT GROUP STATE

  -- IDLE
  if pio.jg.arm_fg.state == pio.jgs.IDLE then
    pio.jg.arm_fg.anim = pio.poseanim.idle
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- DIE
  elseif pio.jg.arm_fg.state == pio.jgs.DIE then
    pio.jg.arm_fg.anim = pio.poseanim.die
    pio.pose_anim_advance(pio.jg.arm_fg, false)
  -- SLACK
  elseif pio.jg.arm_fg.state == pio.jgs.SLACK then
    pio.jg.arm_fg.anim = pio.poseanim.slack
  -- BRACE
  elseif pio.jg.arm_fg.state == pio.jgs.BRACE then
    pio.jg.arm_fg.anim = pio.poseanim.brace
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- RIDE
  elseif pio.jg.arm_fg.state == pio.jgs.RIDE then
    pio.jg.arm_fg.anim = pio.poseanim.ride
    pio.pose_anim_advance(pio.jg.arm_fg, false)
  -- STEP
  elseif pio.jg.arm_fg.state == pio.jgs.STEP then
    pio.jg.arm_fg.anim = pio.poseanim.step
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- WALK
  elseif pio.jg.arm_fg.state == pio.jgs.WALK then
    pio.jg.arm_fg.anim = pio.poseanim.walk
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- RUN
  elseif pio.jg.arm_fg.state == pio.jgs.RUN then
    pio.jg.arm_fg.anim = pio.poseanim.run
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- CLIMB
  elseif pio.jg.arm_fg.state == pio.jgs.CLIMB then
    pio.jg.arm_fg.anim = pio.poseanim.climb
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- CRAWL
  elseif pio.jg.arm_fg.state == pio.jgs.CRAWL then
    pio.jg.arm_fg.anim = pio.poseanim.crawl
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- GET UP BACK
  elseif pio.jg.arm_fg.state == pio.jgs.GET_UP_BACK then
    pio.jg.arm_fg.anim = pio.poseanim.get_up_back
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- GET UP FRONT
  elseif pio.jg.arm_fg.state == pio.jgs.GET_UP_FRONT then
    pio.jg.arm_fg.anim = pio.poseanim.get_up_front
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- GO PRONE
  elseif pio.jg.arm_fg.state == pio.jgs.GO_PRONE then
    pio.jg.arm_fg.anim = pio.poseanim.go_prone
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- TEETER
  elseif pio.jg.arm_fg.state == pio.jgs.TEETER then
    pio.jg.arm_fg.anim = pio.poseanim.teeter
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- PREJUMP
  elseif pio.jg.arm_fg.state == pio.jgs.PREJUMP then
    pio.jg.arm_fg.anim = pio.poseanim.prejump
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- JUMP
  elseif pio.jg.arm_fg.state == pio.jgs.JUMP then
    pio.jg.arm_fg.anim = pio.poseanim.jump
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- POSTJUMP
  elseif pio.jg.arm_fg.state == pio.jgs.POSTJUMP then
    pio.jg.arm_fg.anim = pio.poseanim.postjump
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- FALL
  elseif pio.jg.arm_fg.state == pio.jgs.FALL then
    pio.jg.arm_fg.anim = pio.poseanim.fall
    -- Sync with the legs' animation
    pio.jg.arm_fg.anim_num = pio.jg.legs.anim_num
  -- FLY
  elseif pio.jg.arm_fg.state == pio.jgs.FLY then
    pio.jg.arm_fg.anim = pio.poseanim.fly
  -- SKID
  elseif pio.jg.arm_fg.state == pio.jgs.SKID then
    pio.jg.arm_fg.anim = pio.poseanim.skid
    if pio.pose_anim_advance(pio.jg.arm_fg, false) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.IDLE)
    end
  -- SLIDE
  elseif pio.jg.arm_fg.state == pio.jgs.SLIDE then
    pio.jg.arm_fg.anim = pio.poseanim.slide
    if pio.pose_anim_advance(pio.jg.arm_fg, false) then
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.IDLE)
    end
  -- GRAB TOOL
  elseif pio.jg.arm_fg.state == pio.jgs.GRAB_TOOL then
    pio.jg.arm_fg.anim = pio.poseanim.grab_tool
    if pio.pose_anim_advance(pio.jg.arm_fg, false) then
-- TODO: Move to the next state!
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.IDLE)
    end
  -- DRAW TOOL
  elseif pio.jg.arm_fg.state == pio.jgs.DRAW_TOOL then
    pio.jg.arm_fg.anim = pio.poseanim.draw_tool
    if pio.pose_anim_advance(pio.jg.arm_fg, false) then
-- TODO: Move to the next state!
      pio.set_jg_state(pio.jg.arm_fg, pio.jgs.IDLE)
    end
  -- AIM TOOL AT
  elseif pio.jg.arm_fg.state == pio.jgs.AIM_TOOL_AT then
    pio.jg.arm_fg.anim = pio.poseanim.none
    -- Procedurally aim
  -- POINT AT
  elseif pio.jg.arm_fg.state == pio.jgs.POINT_AT then
    pio.jg.arm_fg.anim = pio.poseanim.point_at
  -- ATOMIZE
  elseif pio.jg.arm_fg.state == pio.jgs.ATOMIZE then
    pio.jg.arm_fg.anim = pio.poseanim.point_at
    pio.jg.arm_fg.point_target = pio:GetAtomizedPos()
  -- ASSEMBLE
  elseif pio.jg.arm_fg.state == pio.jgs.ASSEMBLE then
    pio.jg.arm_fg.anim = pio.poseanim.point_at
    pio.jg.arm_fg.point_target = pio:GetAssembledPos()
  end

  ---------------------------------
  -- ARM BG JOINT GROUP STATE

  -- IDLE
  if pio.jg.arm_bg.state == pio.jgs.IDLE then
    pio.jg.arm_bg.anim = pio.poseanim.idle
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- DIE
  elseif pio.jg.arm_bg.state == pio.jgs.DIE then
    pio.jg.arm_bg.anim = pio.poseanim.die
    pio.pose_anim_advance(pio.jg.arm_bg, false)
  -- SLACK
  elseif pio.jg.arm_bg.state == pio.jgs.SLACK then
    pio.jg.arm_bg.anim = pio.poseanim.slack
  -- BRACE
  elseif pio.jg.arm_bg.state == pio.jgs.BRACE then
    pio.jg.arm_bg.anim = pio.poseanim.brace
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- RIDE
  elseif pio.jg.arm_bg.state == pio.jgs.RIDE then
    pio.jg.arm_bg.anim = pio.poseanim.ride
    pio.pose_anim_advance(pio.jg.arm_bg, false)
  -- STEP
  elseif pio.jg.arm_bg.state == pio.jgs.STEP then
    pio.jg.arm_bg.anim = pio.poseanim.step
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- WALK
  elseif pio.jg.arm_bg.state == pio.jgs.WALK then
    pio.jg.arm_bg.anim = pio.poseanim.walk
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- RUN
  elseif pio.jg.arm_bg.state == pio.jgs.RUN then
    pio.jg.arm_bg.anim = pio.poseanim.run
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- CLIMB
  elseif pio.jg.arm_bg.state == pio.jgs.CLIMB then
    pio.jg.arm_bg.anim = pio.poseanim.climb
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- CRAWL
  elseif pio.jg.arm_bg.state == pio.jgs.CRAWL then
    pio.jg.arm_bg.anim = pio.poseanim.crawl
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- GET UP BACK
  elseif pio.jg.arm_bg.state == pio.jgs.GET_UP_BACK then
    pio.jg.arm_bg.anim = pio.poseanim.get_up_back
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- GET UP FRONT
  elseif pio.jg.arm_bg.state == pio.jgs.GET_UP_FRONT then
    pio.jg.arm_bg.anim = pio.poseanim.get_up_front
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- GO PRONE
  elseif pio.jg.arm_bg.state == pio.jgs.GO_PRONE then
    pio.jg.arm_bg.anim = pio.poseanim.go_prone
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- TEETER
  elseif pio.jg.arm_bg.state == pio.jgs.TEETER then
    pio.jg.arm_bg.anim = pio.poseanim.teeter
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- PREJUMP
  elseif pio.jg.arm_bg.state == pio.jgs.PREJUMP then
    pio.jg.arm_bg.anim = pio.poseanim.prejump
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- JUMP
  elseif pio.jg.arm_bg.state == pio.jgs.JUMP then
    pio.jg.arm_bg.anim = pio.poseanim.jump
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- POSTJUMP
  elseif pio.jg.arm_bg.state == pio.jgs.POSTJUMP then
    pio.jg.arm_bg.anim = pio.poseanim.postjump
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- FALL
  elseif pio.jg.arm_bg.state == pio.jgs.FALL then
    pio.jg.arm_bg.anim = pio.poseanim.fall
    -- Sync with the legs' animation
    pio.jg.arm_bg.anim_num = pio.jg.legs.anim_num
  -- FLY
  elseif pio.jg.arm_bg.state == pio.jgs.FLY then
    pio.jg.arm_bg.anim = pio.poseanim.fly
  -- SKID
  elseif pio.jg.arm_bg.state == pio.jgs.SKID then
    pio.jg.arm_bg.anim = pio.poseanim.skid
    if pio.pose_anim_advance(pio.jg.arm_bg, false) then
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.IDLE)
    end
  -- SLIDE
  elseif pio.jg.arm_bg.state == pio.jgs.SLIDE then
    pio.jg.arm_bg.anim = pio.poseanim.slide
    if pio.pose_anim_advance(pio.jg.arm_bg, false) then
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.IDLE)
    end
  -- GRAB TOOL
  elseif pio.jg.arm_bg.state == pio.jgs.GRAB_TOOL then
    pio.jg.arm_bg.anim = pio.poseanim.grab_tool
    if pio.pose_anim_advance(pio.jg.arm_bg, false) then
-- TODO: Move to the next state!
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.IDLE)
    end
  -- DRAW TOOL
  elseif pio.jg.arm_bg.state == pio.jgs.DRAW_TOOL then
    pio.jg.arm_bg.anim = pio.poseanim.draw_tool
    if pio.pose_anim_advance(pio.jg.arm_bg, false) then
-- TODO: Move to the next state!
      pio.set_jg_state(pio.jg.arm_bg, pio.jgs.IDLE)
    end
  -- AIM TOOL AT
  elseif pio.jg.arm_bg.state == pio.jgs.AIM_TOOL_AT then
    pio.jg.arm_bg.anim = pio.poseanim.none
    -- Procedurally aim
  -- POINT AT
  elseif pio.jg.arm_bg.state == pio.jgs.POINT_AT then
    pio.jg.arm_bg.anim = pio.poseanim.point_at
  -- ATOMIZE
  elseif pio.jg.arm_bg.state == pio.jgs.ATOMIZE then
    pio.jg.arm_bg.anim = pio.poseanim.point_at
    pio.jg.arm_bg.point_target = pio:GetAtomizedPos()
  -- ASSEMBLE
  elseif pio.jg.arm_bg.state == pio.jgs.ASSEMBLE then
    pio.jg.arm_bg.anim = pio.poseanim.point_at
    pio.jg.arm_bg.point_target = pio:GetAssembledPos()
  end

  ---------------------------------
  -- NECK JOINT GROUP STATE

  -- IDLE
  if pio.jg.neck.state == pio.jgs.IDLE then
    pio.jg.neck.anim = pio.poseanim.idle
    pio.pose_anim_advance(pio.jg.neck, false)
  -- DIE
  elseif pio.jg.neck.state == pio.jgs.DIE then
    pio.jg.neck.anim = pio.poseanim.die
    pio.pose_anim_advance(pio.jg.neck, false)
  -- SLACK
  elseif pio.jg.neck.state == pio.jgs.SLACK then
    pio.jg.neck.anim = pio.poseanim.slack
  -- BRACE
  elseif pio.jg.neck.state == pio.jgs.BRACE then
    pio.jg.neck.anim = pio.poseanim.brace
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- RIDE
  elseif pio.jg.neck.state == pio.jgs.RIDE then
    pio.jg.neck.anim = pio.poseanim.ride
    pio.pose_anim_advance(pio.jg.neck, false)
  -- STEP
  elseif pio.jg.neck.state == pio.jgs.STEP then
    pio.jg.neck.anim = pio.poseanim.step
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- WALK
  elseif pio.jg.neck.state == pio.jgs.WALK then
    pio.jg.neck.anim = pio.poseanim.walk
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- RUN
  elseif pio.jg.neck.state == pio.jgs.RUN then
    pio.jg.neck.anim = pio.poseanim.run
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- CLIMB
  elseif pio.jg.neck.state == pio.jgs.CLIMB then
    pio.jg.neck.anim = pio.poseanim.climb
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- CRAWL
  elseif pio.jg.neck.state == pio.jgs.CRAWL then
    pio.jg.neck.anim = pio.poseanim.crawl
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- GET UP BACK
  elseif pio.jg.neck.state == pio.jgs.GET_UP_BACK then
    pio.jg.neck.anim = pio.poseanim.get_up_back
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- GET UP FRONT
  elseif pio.jg.neck.state == pio.jgs.GET_UP_FRONT then
    pio.jg.neck.anim = pio.poseanim.get_up_front
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- GO PRONE
  elseif pio.jg.neck.state == pio.jgs.GO_PRONE then
    pio.jg.neck.anim = pio.poseanim.go_prone
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- TEETER
  elseif pio.jg.neck.state == pio.jgs.TEETER then
    pio.jg.neck.anim = pio.poseanim.teeter
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- PREJUMP
  elseif pio.jg.neck.state == pio.jgs.PREJUMP then
    pio.jg.neck.anim = pio.poseanim.prejump
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- JUMP
  elseif pio.jg.neck.state == pio.jgs.JUMP then
    pio.jg.neck.anim = pio.poseanim.jump
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- POSTJUMP
  elseif pio.jg.neck.state == pio.jgs.POSTJUMP then
    pio.jg.neck.anim = pio.poseanim.postjump
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- FALL
  elseif pio.jg.neck.state == pio.jgs.FALL then
    pio.jg.neck.anim = pio.poseanim.fall
    -- Sync with the legs' animation
    pio.jg.neck.anim_num = pio.jg.legs.anim_num
  -- FLY
  elseif pio.jg.neck.state == pio.jgs.FLY then
    pio.jg.neck.anim = pio.poseanim.fly
  -- SKID
  elseif pio.jg.neck.state == pio.jgs.SKID then 
    pio.jg.neck.anim = pio.poseanim.skid
    if pio.pose_anim_advance(pio.jg.neck, false) then
-- TODO: Sync with arm animations?
      pio.set_jg_state(pio.jg.neck, pio.jgs.IDLE)
    end
  -- SLIDE
elseif pio.jg.neck.state == pio.jgs.SLIDE then
    pio.jg.neck.anim = pio.poseanim.slide
    if pio.pose_anim_advance(pio.jg.neck, false) then
-- TODO: Sync with arm animations?
      pio.set_jg_state(pio.jg.neck, pio.jgs.IDLE)
    end
  -- GRAB TOOL
  elseif pio.jg.neck.state == pio.jgs.GRAB_TOOL then
    pio.jg.neck.anim = pio.poseanim.grab_tool
    if pio.pose_anim_advance(pio.jg.neck, false) then
-- TODO: Sync with arm animations?
      pio.set_jg_state(pio.jg.neck, pio.jgs.IDLE)
    end
  -- DRAW TOOL
  elseif pio.jg.neck.state == pio.jgs.DRAW_TOOL then
    pio.jg.neck.anim = pio.poseanim.draw_tool
    if pio.pose_anim_advance(pio.jg.neck, false) then
-- TODO: Sync with arm animations?
      pio.set_jg_state(pio.jg.neck, pio.jgs.IDLE)
    end
  -- AIM TOOL AT
  elseif pio.jg.neck.state == pio.jgs.AIM_TOOL_AT then
    pio.jg.neck.anim = pio.poseanim.none
    -- Procedurally aim
  -- POINT AT
  elseif pio.jg.neck.state == pio.jgs.POINT_AT then
    pio.jg.neck.anim = pio.poseanim.point_at
  -- ATOMIZE
  elseif pio.jg.neck.state == pio.jgs.ATOMIZE then
    pio.jg.neck.anim = pio.poseanim.point_at
    pio.jg.neck.point_target = pio:GetAtomizedPos()
  -- ASSEMBLE
  elseif pio.jg.neck.state == pio.jgs.ASSEMBLE then
    pio.jg.neck.anim = pio.poseanim.point_at
    pio.jg.neck.point_target = pio:GetAssembledPos()
  end


  ---------------------------------------------------------------
  -- Apply friction values
  ---------------------------------------------------------------

  if pio.is_ambulating_now() then
    pio:MOSet("Torso"):SetSlippery(0.85);
    pio:MOSet("Head"):SetSlippery(0.85);
    pio:MOSet("Backpack"):SetSlippery(0.85);
    
    if (pio.jg.legs.anim_num == 3) or (pio.jg.legs.anim_num == 4) or (pio.jg.legs.anim_num == 5) then 
      pio:MOSet("Leg FG"):SetSlippery(0)
      pio:MOSet("Leg BG"):SetSlippery(1)
    else 
      pio:MOSet("Leg FG"):SetSlippery(1)
      pio:MOSet("Leg BG"):SetSlippery(0)
    end
    pio:MOSet("Arm FG"):SetSlippery(1);
    pio:MOSet("Arm BG"):SetSlippery(1);
  -- When just standing with balance
  elseif pio.is_balancing_now() then
    pio:MOSet("Torso"):SetSlippery(pio.slippery_body);
    pio:MOSet("Head"):SetSlippery(pio.slippery_body);

    pio:MOSet("Leg FG"):SetSlippery(pio.slippery_limbs);
    pio:MOSet("Leg BG"):SetSlippery(pio.slippery_limbs);
    pio:MOSet("Arm FG"):SetSlippery(pio.slippery_limbs);
    pio:MOSet("Arm BG"):SetSlippery(pio.slippery_limbs);
  -- Sliding or flying now
  elseif pio.is_sliding_now() or pio.is_flying_now() or pio.is_crawling_now() then
    pio:MOSet(""):SetSlippery(0.85)
  -- Slack and in a heap, high friction so comes to stop fast
  else
    pio:MOSet(""):SetSlippery(0.25)
  end


  -----------------------------------------------------------
  -- Apply the currently selected poses to the joint groups
  -----------------------------------------------------------

  pio.fade_to_current_pose(pio.jg.legs)
  pio.fade_to_current_pose(pio.jg.arm_fg)
  pio.fade_to_current_pose(pio.jg.arm_bg)
  pio.fade_to_current_pose(pio.jg.neck)


  -----------------------------------------------------------
  -- Make arms and head point at set targets, if applicable
  -----------------------------------------------------------

  if pio.jg.arm_fg.anim == pio.poseanim.point_at then
    -- Turn the shoulder joint so the arm aims at target
    pio.aim_joint_at_target(pio:GetJointByName("Arm FG Shoulder"), pio.jg.arm_fg.point_target, math.pi / 10)
  end
  if pio.jg.arm_bg.anim == pio.poseanim.point_at then
    -- Turn the shoulder joint so the arm aims at target
    pio.aim_joint_at_target(pio:GetJointByName("Arm BG Shoulder"), pio.jg.arm_bg.point_target, math.pi / 10)
  end
  if pio.jg.neck.anim == pio.poseanim.point_at then
-- TODO: FIX that head orientation all round the planetoid
    -- Turn the neck joint so the head looks at target 
    local joint = pio:GetJointByName("Neck")
    if joint then
      local target_vec = pio.jg.neck.point_target - joint:GetWorldPositionA()
      target_vec = pio.body_mo:UnRotateVectorForGravity(target_vec)
      -- Don't crane neck to look for a thing behind the player
      if (pio:IsMirrored() and target_vec.mX < 0) or (not pio:IsMirrored() and target_vec.mX > 0) then
        local trim_angle = 0
        if pio:IsMirrored() then trim_angle = math.pi * 1.0 end
        joint:FadeAngularSpring(-target_vec:GetAngleRad() + trim_angle, 0)
      end
    end
  end
  

  -----------------------------------------------------------
  -- Apply balancers to body
  -----------------------------------------------------------
    
  -- Balacing on two legs
  if pio.balance_state == pio.balance.BIPED then
    -- Activate balancing
    pio:MOSet("Torso"):SetBalancer(true)
    pio:MOSet("Head"):SetBalancer(pio.jg.neck.anim ~= pio.poseanim.point_at)
    -- Set the power
    pio:MOSet("Torso"):SetBalancerPower(pio.balance_power * 1.0)
		-- Head should have no balancer in idle balance mode
    pio:MOSet("Head"):SetBalancerPower(0)
    
    -- Set angle
    
    -- Ambulating lean
    if pio.is_ambulating_now() then
      if pio.is_ambulating_left then
        if pio:IsMirrored() then
          pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(-pio.balance_lean)) 
        else
          pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(0))
        end
      else
        if not pio:IsMirrored() then
          pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(pio.balance_lean))
        else
          pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(0))
        end
      end
    -- Simple lean
    else
      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(pio.balance_lean))
    end
		
  -- Get up off the deck
  elseif pio.balance_state == pio.balance.GET_UP then
    pio:MOSet("Torso"):SetBalancer(true)
    pio:MOSet("Head"):SetBalancer(pio.jg.neck.anim ~= pio.poseanim.point_at)
    pio:MOSet("Torso"):SetBalancerPower(pio.balance_power * 0.2)
    pio:MOSet("Head"):SetBalancerPower(pio.balance_power * 0.25)
    
    
  -- Pull body down to the deck
  elseif pio.balance_state == pio.balance.GO_PRONE and pio.get_altitude(2) < 1.5 then
    pio:MOSet("Torso"):SetBalancer(true)
    pio:MOSet("Head"):SetBalancer(false)
    pio:MOSet("Torso"):SetBalancerPower(pio.balance_power)
    pio:MOSet("Head"):SetBalancerPower(pio.balance_power * 0.8)
    
    if pio:IsMirrored() then
      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(-math.pi / 2))
    else
      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(math.pi / 2))
    end
  
  
  -- Flight control magical spring assistance
elseif pio.balance_state == pio.balance.FLYING then
    -- Steer with your head when flying, more dynamic
    pio:MOSet("Torso"):SetBalancer(true)
    pio:MOSet("Head"):SetBalancer(pio.jg.neck.anim ~= pio.poseanim.point_at)
    -- Have a stronger spring on body in beginning of boost, to counter the lean by Jet PS boom
    if not pio.total_flight_timer:Elapsed(0.25) then
      pio:MOSet("Torso"):SetBalancerPower(pio.balance_power * 10.0)
    else
      pio:MOSet("Torso"):SetBalancerPower(pio.balance_power * 0.25)
    end
    pio:MOSet("Head"):SetBalancerPower(pio.balance_power)
    
    -- Look at player input and get the vector to alter the balance vector by
    local controlVector = controller:MoveVector()
    controlVector:Invert()
    local target_angle = pio.head_mo:GetBalanceToAngle()
    
    
    -- RELATIVE CONTROL
    if pio.relative_flight_control then
      -- The speed at which input affects the target angle
      local input_speed = 0.05
      -- Retard the input speed when in beginning of flight
--      if not pio.total_flight_timer:Elapsed(0.5) then
--        input_speed = 0
--      elseif not pio.total_flight_timer:Elapsed(1.0) then
        input_speed = input_speed * 0.6
--      end
      -- Use head to steer in air (where the head goes, the body follows)
      target_angle = pio.head_mo:GetBalanceToAngle() + ((-controlVector:GetAngleRad() - math.halfpi) * input_speed)
    else
      -- ABSOLUTE CONTROL
      -- The extra fourthpi needed because head tilt
      if controlVector:GetLength() > 0.1 and pio.total_flight_timer:Elapsed(0.5) then
        local flip_head_adjust = -math.fourthpi
        if pio:IsMirrored() then flip_head_adjust = math.fourthpi end
        target_angle = pio.body_mo:RotateAngleForGravity(-controlVector:GetAngleRad() - math.halfpi + flip_head_adjust)
      end
    end
    
--[[
    -- Don't let the target angle get too far away from the actual angle of the body orientation    
--    local head_angle = math.frotate(-math.pi, math.pi, pio.head_mo:GetOrientation())
--    target_angle = math.frotate(-math.pi, math.pi, target_angle)
    local head_angle = pio.head_mo:GetOrientation() - math.halfpi
    
    local angle_delta = head_angle - target_angle
    if angle_delta > math.halfpi then 
      target_angle = head_angle + math.halfpi
    elseif angle_delta < -math.halfpi then
      target_angle = head_angle - math.halfpi
    end
]]--
    -- Note that in the flipping handling in input handling the BalanceToAngle gets adjusted on flip to match what it was before the flip
    pio:MOSet("Torso"):SetBalanceToAngle(target_angle)
    pio:MOSet("Head"):SetBalanceToAngle(target_angle)
    
--    pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(-controlVector:GetAngleRad() - (math.pi / 2)))
--    pio:MOSet("Head"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(-controlVector:GetAngleRad() - (math.pi / 2)))
    
    -- TEMP DEBUG
    local balance_vector = VectorF(1.0, 0)
    balance_vector:RotateRight(pio.head_mo:GetBalanceToAngle() - math.halfpi)
    pio.head_mo:DrawWireVector(pio:GetApproxPos(), balance_vector, ColorRGBA(100,255,100,255))
    
    local head_vector = VectorF(1.0, 0)
    head_vector:RotateLeft(pio.head_mo:GetOrientation() + math.halfpi)
    pio.head_mo:DrawWireVector(pio:GetApproxPos(), head_vector, ColorRGBA(100,100,255,255))
    
    
  -- Crawling or climbing
  elseif pio.balance_state == pio.balance.PRONE then
    pio:MOSet("Torso"):SetBalancer(false)
    pio:MOSet("Head"):SetBalancer(false)
    pio:MOSet("Torso"):SetBalancerPower(pio.balance_power * 0.5)
    pio:MOSet("Head"):SetBalancerPower(pio.balance_power * 0.25)
    
    if pio:IsMirrored() then
      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(-math.pi / 2))
    else
      pio:MOSet("Torso"):SetBalanceToAngle(pio.body_mo:RotateAngleForGravity(math.pi / 2))
    end
    
    
  -- Not balancing! 
  else
    pio:MOSet("Torso"):SetBalancer(false)
    pio:MOSet("Head"):SetBalancer(false)
  end

--[[
  -- TODO: CLEAN UP BELOW
  -- At which velocity we lose our footing
  local lose_balance_velocity = 10

  if pio.body_mo then
    -- Check if we should still be in balance - can't lose balance due to mirroring velocities
    if not pio.mirror_change_timer:Elapsed(0.1) or
       (math.abs(pio.body_mo:GetVelocity().mX) < lose_balance_velocity and
        math.abs(pio.body_mo:GetVelocity().mY) < lose_balance_velocity) or
       pio.flying_now then
      -- Yep, we can be balanced now
      pio:MOSet("Torso"):SetBalancer(pio.is_ambulating_now())
    else
      -- Lose balance
      pio.idle()
    --    pio.body_mo:SetSlippery(0.25)
    end
  end

]]--
  -----------------------------------------------------------
  -- Apply locomotion forces to body
  -----------------------------------------------------------

  if pio.is_ambulating_now() then -- and pio.balance_state == pio.balance.BIPED then  
    if pio.legs_colliding_with_something() or pio.arms_colliding_with_something() then -- now touching the ground
      -- Left/right push force
      local push_lr = 300
      -- Push upward to get over obstacles and to just stay on the feet
      local push_up = 200
      -- Max running speed
      local max_run_vel = 7
      -- Max climbing speed
      local max_climb_vel = 2
      
      -- If moving backwards, push a lot less so the feet can keep up
      if (pio.is_ambulating_left and not pio:IsMirrored()) or (not pio.is_ambulating_left and pio:IsMirrored()) then
        push_lr = 150
        max_run_vel = 5
      end
            
      -- If body is tilted over a lot from the upright 'balanced state' then make the push less powerful, so he can't lay on his back or belly and run forward at top speed
  --[[      local tilt_angle = math.abs(pio.body_mo:GetOrientation() - pio.body_mo:GetBalanceToAngle())
      -- If we're beyond fallen over, don't push at all
      if tilt_angle > (math.pi / 2) then
        push_lr = 0
      -- If we're not quite horizontal, then retard the psuh forces according to how close we are to horizontal
      else
        if tilt_angle > 0.5 then push_lr = push_lr * (1.0 - ((tilt_angle - 0.5) / ((math.pi / 2) - 0.5))) end
      end
  ]]
      -- If RUNNING, don't push up quite as much
      if pio.jg.legs.anim == pio.poseanim.run then
        push_up = 100
        max_climb_vel = 1
      -- If WALKING, don't go as fast
      elseif pio.jg.legs.anim == pio.poseanim.walk then
        push_lr = 175
        max_run_vel = 2.5
        push_up = 130
        max_climb_vel = 1.5
      -- If CLIMBING or CRAWLING, don't push sideways so much
      elseif pio.jg.legs.anim == pio.poseanim.climb then
        push_lr = 150
        max_run_vel = 3.5
        push_up = 100
        max_climb_vel = 1.0
      elseif pio.jg.legs.anim == pio.poseanim.crawl then
        push_lr = 150
        max_run_vel = 3.5
        push_up = 100
        max_climb_vel = 1.0
      end
      
      -- Actually apply the forces, but only if we're going under the top speed both forward and up
      if pio.is_ambulating_left then
        if pio.unrotated_vel.mX > -max_run_vel then
          pio.body_mo:PullTest4(pio.body_mo:RotateVectorForGravity(VectorF(-push_lr, 0)))
        end
      else 
        if pio.unrotated_vel.mX < max_run_vel then
          pio.body_mo:PullTest4(pio.body_mo:RotateVectorForGravity(VectorF(push_lr, 0)))
        end
      end
      -- Pull up too to keep him from dragging on ground
      if pio.unrotated_vel.mY < max_climb_vel then
        pio.body_mo:PullTest4(pio.body_mo:RotateVectorForGravity(VectorF(0, push_up)))
      end
    end
  end


  -----------------------------------------------------------
  -- Update particle systems
  -----------------------------------------------------------

  -----------------------------------------------------------
  -- Jetpack


  -----------------------------------------------------------
  -- Skimming effects


  -----------------------------------------------------------
  -- Shield effects
  
  -------------------------------------------------------------
  -- Clear the Joint Group flags showing their state was changed this frame
  
  for name, joint_group in pairs(pio.jg) do
    joint_group.state_changed = false
  end
  
	------------------------------------------------------------
	-- Aurium use and recharge updates
	
	-- Aurium use
	if pio.is_flying_now() then
-- TODO: Figure out some procedural and understandable way to know what an appropriate amount of A charge a specfic thing uses
		pio.aurium_charge = pio.aurium_charge - (pio.flight_aurium_consumption * scene:GetDeltaTime())
		-- Reset the timer since last A change
		pio.aurium_charge_change_timer:Reset()
		-- Cap at 0
		if pio.aurium_charge < 0 then pio.aurium_charge = 0 end
	end
	
	-- Recharge if last Aurium use was past the required delay
	if pio.no_aurium_use_timer:TimeIsUp() and pio.aurium_charge < pio.aurium_capacity then
		-- How much we want to try to consume from the Aurium store in the Controller
		local charge_consumption = pio.aurium_recharge_rate * scene:GetDeltaTime()
		-- Don't try to consume more than we have room for in this battery
		if pio.aurium_capacity - pio.aurium_charge < charge_consumption then charge_consumption = pio.aurium_capacity - pio.aurium_charge end
		-- Ask to consume and also check how much we actually got to consume from the Controller
		charge_consumption = controller:ConsumeAurium(charge_consumption)
		-- Only add to charge if we actually had anything to consume from the Aurium Store in the Controller
		if charge_consumption > 0 then
			-- Add the consumed Aurium charge from the Store to this Assembly
			pio.aurium_charge = pio.aurium_charge + charge_consumption
			-- Reset the timer since last A change
			pio.aurium_charge_change_timer:Reset()
			-- Also reset timer since last A recharge
			pio.aurium_recharge_timer:Reset()
			-- Cap at max
			if pio.aurium_charge > pio.aurium_capacity then pio.aurium_charge = pio.aurium_capacity end
		end
	end

	-- Detect and time since last Controller's Aurium Store value change
	if pio.aurium_last_store ~= controller:GetAurium() then
		pio.aurium_store_change_timer:Reset()
	end
	-- Store last recorded Aurium store value
	pio.aurium_last_store = controller:GetAurium()
	
  ------------------------------------------------------------
  -- Debug draw the JGSs above the head

  if not pio.body_mo or not pio.head_mo then return end
--[[
  -- Joint group states
  local textPos = pio:GetApproxPos() + pio.body_mo:RotateVectorForGravity(VectorF(0, 8))
  scene:WriteToScreenByWorldPos("Legs: " .. pio.jg.legs.anim.name, textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.25))
  scene:WriteToScreenByWorldPos("Arm FG: " .. pio.jg.arm_fg.anim.name, textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.25))
  scene:WriteToScreenByWorldPos("Arm BG: " .. pio.jg.arm_bg.anim.name, textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.25))
  scene:WriteToScreenByWorldPos("Neck: " .. pio.jg.neck.anim.name, textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  
  -- Leg anim number
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.25))
  scene:WriteToScreenByWorldPos("Legs anim number: " .. pio.jg.legs.anim_num, textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  
  -- Biped balancing state
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.35))
  if pio.balance_state == pio.balance.BIPED then
    scene:WriteToScreenByWorldPos("Biped Balance!", textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  else
    scene:WriteToScreenByWorldPos("No Balance!", textPos, VectorF(0,0), Yes, ColorRGBA(255,100,100,255))
  end
  
  -- Slippery of legs
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.35))
  scene:WriteToScreenByWorldPos("Body slippery: " .. tostring(pio.body_mo:GetSlippery()), textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  
  -- Slippery of head
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.35))
  scene:WriteToScreenByWorldPos("Head slippery: " .. tostring(pio.head_mo:GetSlippery()), textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  

  -- Altitude
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.35))
  local altitude = pio.get_altitude(10)
  if altitude then
    scene:WriteToScreenByWorldPos("Altitude: " .. tostring(altitude), textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  else
    scene:WriteToScreenByWorldPos("Altitude over 100m!", textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
  end
]]--
--[[
  -- Angle of the move vector
  local moveAngle = controller:MoveVector():GetAngleRad()--pio.body_mo:UnRotateAngleForGravity(pio.body_mo:GetOrientation())
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.35))
  scene:WriteToScreenByWorldPos("Move input angle: " .. tostring(moveAngle), textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
 
  -- Angle of the torso
  local bodyAngle = pio.body_mo:RotateAngleForGravity(pio.body_mo:GetOrientation())
  textPos = textPos - pio.body_mo:RotateVectorForGravity(VectorF(0, 0.35))
  scene:WriteToScreenByWorldPos("Body angle: " .. tostring(bodyAngle), textPos, VectorF(0,0), Yes, ColorRGBA(100,255,100,255))
]]--

  -- Update resource bars etc in UI
  pioneer_ui.update(pio)
	
end












































